/* $Id: manager.c,v 1.4 2001/11/27 22:09:05 eric Exp $ */

/* SUMMARY
 *
 * manager.c - manager
 *
 * The program manager acts as an intermediary between local PJVM 
 * utilities and remote JVMs under the control of pjvms.  The local
 * manager connects to pjvms remotely, and then forwards all requests 
 * to either the remote PJVM Server, or directly to a remote JVM,
 * depending on the command.
 *
 *
 * REVISION HISTORY
 *
 * $Log: manager.c,v $
 * Revision 1.4  2001/11/27 22:09:05  eric
 * Added support for listobjects
 *
 * Revision 1.3  2001/11/14 18:13:06  eric
 * Corrected error which wrote to the wrong buffer when invalid JVM was selected in listmethods
 *
 * Revision 1.2  2001/11/04 05:29:04  eric
 * Added listmethods
 *
 * Revision 1.1  2001/10/23 21:30:02  eric
 * Initial revision
 *
 */

#define _USE_BSD
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>

#include "errexit.h"
#include "passiveTCP.h"
#include "connectTCP.h"
#include "strtok2.h"

#define QLEN 32         /* maximum connection count */
#define MAX_SERVERS 250  /* maximum number of concurrent JVM servers */
#define BUFSIZE 4096
#define LINELEN 128
#define REMOTE 0        /* constants for loadclass source */
#define LOCAL 1
#define UNDEFINED 5
#define TRUE 1          /* constants for listclass source */
#define FALSE 0

struct JVM_server {
   char host[ LINELEN + 1 ];
   int port_number;
};

extern int errno;
void manager( int slave_socket, char *host, char *service,
              struct JVM_server JVM_servers[],  int *server_index );


/*  Function main()
 *  --------------------------------------------------------------------
 *
 *    Iterative TCP server for local PJVM connections.
 *
 */

int main( int argc, char *argv[] )
{
   char host[ LINELEN + 1], service[ LINELEN + 1],
        local_service[ LINELEN + 1];
   struct sockaddr_in client_address;
   int master_socket, slave_socket, c, server_index = 0;
   unsigned int address_length;
   struct JVM_server *JVM_servers;

   JVM_servers = malloc( sizeof( struct JVM_server ) * MAX_SERVERS );
   if ( JVM_servers == NULL )
      errexit( "malloc: %s\n", strerror( errno ) );

   strcpy( host, "localhost" );
   strcpy( service, "pjvms" );
   strcpy( local_service, "pjvmmanager" );

   /* disable getopt() error messages */
   opterr = 0;

   while ( ( c = getopt( argc, argv, "h:s:l:" ) ) != -1 )
   {
      switch ( c )
      {
         case 'h': if ( strlen( optarg ) != 0 )
	              strncpy( host, optarg, LINELEN );
                   break;
         case 's': if ( strlen( optarg ) != 0 )
	              strncpy( service, optarg, LINELEN );
                   break;
         case 'l': if ( strlen( optarg ) != 0 )
	              strncpy( local_service, optarg, LINELEN );
                   break;
         case '?': errexit( "Usage: manager [-h hostname] "
	                    "[-s remote service] "
			    "[-l local service]\n" );
      }
   }

   /* creates a passive socket to accept connection from clients */
   master_socket = passiveTCP( local_service, QLEN );

   for ( ;; )
   {
      /* open a new socket to deal with the incoming connection */
      address_length = sizeof( client_address );
      slave_socket = accept( master_socket, ( struct sockaddr * )
                             &client_address, &address_length );
      
      if ( slave_socket < 0 )
         errexit( "accept failed: %s\n", strerror( errno ) );

      manager( slave_socket, host, service, JVM_servers, 
               &server_index );

      ( void ) close( slave_socket );
   }
}


/*  Function manager()
 *  --------------------------------------------------------------------
 *
 *    Provide local access to a remote PJVM Server
 *
 */

void manager( int slave_socket, char *host, char *service,
            struct JVM_server JVM_servers[],  int *server_index )
{
   char buf[ LINELEN + 1], largebuf[ BUFSIZE + 1 ], 
        secondbuf[ LINELEN + 1 ], *pointer = buf, delimiter;
   int inchars, cc, n, s, i, jvm, jvm_selected = FALSE;

   for ( cc = 0; cc < LINELEN; cc += n )
   {
      n = read( slave_socket, buf, LINELEN );
      if ( n < 0 )
         errexit( "read: %s\n", strerror( errno ) );
   }

   /* if command comes from requestjvm */

   if ( strncmp( buf, "jvm", 3 ) == 0 )
   {
      s = connectTCP( host, service );

      ( void ) write( s, buf, LINELEN );

      for ( inchars = 0; inchars < LINELEN; inchars += n )
      {
         n = read( s, buf, LINELEN );
         if ( n < 0 )
            errexit( "read: %s\n", strerror( errno ) );
      }

      jvm = atoi( buf );
      
      for ( inchars = 0; inchars < LINELEN; inchars += n )
      {
         n = read( s, buf, LINELEN );
         if ( n < 0 )
            errexit( "read: %s\n", strerror( errno ) );
      }

      if ( jvm == 0 )
      {
         strcpy( ( JVM_servers[ *server_index ] ).host, host );
         ( JVM_servers[ *server_index ] ).port_number = atoi( buf );
	      sprintf( buf, "0 %d %d a", ( *server_index )++, atoi( buf ) );
      }
      else
         sprintf( buf, "%d", jvm );
      
      ( void ) write( slave_socket, buf, LINELEN );

      return;
   }

   /* if command comes from listjvm */

   else if ( strncmp( buf, "lsj", 3 ) == 0 )
   {
      sprintf( buf, "%d", *server_index );
      ( void ) write( slave_socket, buf, LINELEN );

      /* if there are no JVMs, return now */

      if ( *server_index == 0 )
         return;


      sprintf( largebuf, "JVM\tHost\t\tPort\n" );

      for ( n = 0; ( n < *server_index ) && 
                        ( strlen( largebuf ) < BUFSIZE ); n++ )
      {
         sprintf( buf, "%d\t%s\t%d\n", n, JVM_servers[ n ].host,
                  JVM_servers[ n ].port_number );
         strncat( largebuf, buf, BUFSIZE - strlen( largebuf ) );
      }

      ( void ) write( slave_socket, largebuf, BUFSIZE );

      /* HEY ERIC!!!!!!! - you need to implement more detailed ls
         stuff here - such as which classes loaded, etc. */

      return; /* for now */
   }

   /* if command comes from killjvm */

   else if ( strncmp( buf, "kvm", 3 ) == 0 )
   {
      jvm = atoi( &( buf[ 4 ] ) ); 
      
      if ( ( jvm < 0 ) || ( jvm >= *server_index ) )
      {
         sprintf( buf, "-10000" );
         ( void ) write( slave_socket, buf, LINELEN );
         return;
      }

      sprintf( largebuf, "%d", JVM_servers[ jvm ].port_number );
      s = connectTCP( JVM_servers[ jvm ].host, largebuf );
      ( void ) write( s, buf, LINELEN );

      for ( cc = 0; cc < LINELEN; cc += n )
      {
         n = read( s, buf, LINELEN );
         if ( n < 0 )
            errexit( "read: %s\n", strerror( errno ) );
      }

      ( void ) memmove( &JVM_servers[ jvm ], &JVM_servers[ jvm + 1 ],
                        sizeof( JVM_servers[ jvm ] ) * ( *server_index -
                        jvm ) );
      ( *server_index )--;
      ( void ) write( slave_socket, buf, LINELEN );
      
      return;
   }

   /* if command comes from loadclass */

   else if ( strncmp( buf, "ldc", 3 ) == 0 )
   {
      jvm = atoi( &( buf[ 4 ] ) );

      if ( ( jvm < 0 ) || ( jvm >= *server_index ) )
      {
         sprintf( buf, "-10000" );
         ( void ) write( slave_socket, buf, LINELEN );
         return;
      }

      sprintf( largebuf, "%d", JVM_servers[ jvm ].port_number );
      s = connectTCP( JVM_servers[ jvm ].host, largebuf );
      ( void ) write( s, buf, LINELEN );

      /* discard first 2 tokens of buffer */

      strtok2( &pointer, &delimiter, " " );
      strtok2( &pointer, &delimiter, " " );

      /* just read back the results and pass on, PJVM will take
         care of locating the class */

      for ( cc = 0; cc < LINELEN; cc += n )
      {
         n = read( s, buf, LINELEN );
         if ( n < 0 )
            errexit( "read: %s\n", strerror( errno ) );
      }

      ( void ) write( slave_socket, buf, LINELEN );

      return;
   }

   /* if command comes from listclass */

   else if ( strncmp( buf, "lsc", 3 ) == 0 )
   {
      /* discard first token of buffer */
      strtok2( &pointer, &delimiter, " " );

      jvm_selected = atoi( strtok2( &pointer, &delimiter, " " ) );
      if ( jvm_selected == FALSE )
         jvm = 0;
      else
         jvm = atoi( strtok2( &pointer, &delimiter, " " ) ); 

      if ( ( jvm_selected < 0 ) || ( jvm < 0 ) ||
           ( jvm >= *server_index ) )
      {
         sprintf( buf, "-10000" );
         ( void ) write( slave_socket, buf, LINELEN );
         return;
      }

      if ( jvm_selected == TRUE )
      {
         sprintf( largebuf, "%d", JVM_servers[ jvm ].port_number );
         s = connectTCP( JVM_servers[ jvm ].host, largebuf );
         ( void ) write( s, buf, LINELEN );

         for ( cc = 0; cc < BUFSIZE; cc += n )
         {
            n = read( s, largebuf, BUFSIZE );
            if ( n < 0 )
               errexit( "read: %s\n", strerror( errno ) );
         }

         sprintf( buf, "0" );
         ( void ) write( slave_socket, buf, LINELEN );
         ( void ) write( slave_socket, largebuf, BUFSIZE );
      }

      else
      {
         /* let the client know how many are coming */

         sprintf( secondbuf, "%d", *server_index - 1 );
         ( void ) write( slave_socket, secondbuf, LINELEN );

         for ( i = 0; i < *server_index; i++ )
         {
            sprintf( largebuf, "%d", JVM_servers[ i ].port_number );
            s = connectTCP( JVM_servers[ i ].host, largebuf );
            ( void ) write( s, buf, LINELEN );

            for ( cc = 0; cc < BUFSIZE; cc += n )
            {
               n = read( s, largebuf, BUFSIZE );
               if ( n < 0 )
                  errexit( "read: %s\n", strerror( errno ) );
            }

            ( void ) write( slave_socket, largebuf, BUFSIZE );
         }
      }

      jvm_selected = FALSE;
      return;
   }

   /* if command comes from listobjects */

   else if ( strncmp( buf, "lso", 3 ) == 0 )
   {
      /* discard first token of buffer */
      strtok2( &pointer, &delimiter, " " );

      jvm_selected = atoi( strtok2( &pointer, &delimiter, " " ) );
      if ( jvm_selected == FALSE )
         jvm = 0;
      else
         jvm = atoi( strtok2( &pointer, &delimiter, " " ) ); 

      if ( ( jvm_selected < 0 ) || ( jvm < 0 ) ||
           ( jvm >= *server_index ) )
      {
         sprintf( buf, "-10000" );
         ( void ) write( slave_socket, buf, LINELEN );
         return;
      }

      if ( jvm_selected == TRUE )
      {
         sprintf( largebuf, "%d", JVM_servers[ jvm ].port_number );
         s = connectTCP( JVM_servers[ jvm ].host, largebuf );
         ( void ) write( s, buf, LINELEN );

         for ( cc = 0; cc < BUFSIZE; cc += n )
         {
            n = read( s, largebuf, BUFSIZE );
            if ( n < 0 )
               errexit( "read: %s\n", strerror( errno ) );
         }

         sprintf( buf, "0" );
         ( void ) write( slave_socket, buf, LINELEN );
         ( void ) write( slave_socket, largebuf, BUFSIZE );
      }

      else
      {
         /* let the client know how many are coming */

         sprintf( secondbuf, "%d", *server_index - 1 );
         ( void ) write( slave_socket, secondbuf, LINELEN );

         for ( i = 0; i < *server_index; i++ )
         {
            sprintf( largebuf, "%d", JVM_servers[ i ].port_number );
            s = connectTCP( JVM_servers[ i ].host, largebuf );
            ( void ) write( s, buf, LINELEN );

            for ( cc = 0; cc < BUFSIZE; cc += n )
            {
               n = read( s, largebuf, BUFSIZE );
               if ( n < 0 )
                  errexit( "read: %s\n", strerror( errno ) );
            }

            ( void ) write( slave_socket, largebuf, BUFSIZE );
         }
      }

      jvm_selected = FALSE;
      return;
   }

   /* if command comes from listconstructors */

   else if ( strncmp( buf, "lco", 3 ) == 0 )
   {
      /* discard the first token */
      strtok2( &pointer, &delimiter, " " );

      jvm = atoi( strtok2( &pointer, &delimiter, " " ) );

      if ( ( jvm < 0 ) || ( jvm >= *server_index ) )
      {
         sprintf( buf, "-10000" );
         ( void ) write( slave_socket, buf, LINELEN );
         return;
      }

      sprintf( largebuf, "%d", JVM_servers[ jvm ].port_number );
      s = connectTCP( JVM_servers[ jvm ].host, largebuf );
      ( void ) write( s, buf, LINELEN );

      for ( cc = 0; cc < BUFSIZE; cc += n )
      {
         n = read( s, largebuf, BUFSIZE );
         if ( n < 0 )
            errexit( "read: %s\n", strerror( errno ) );
      }

      ( void ) write( slave_socket, largebuf, BUFSIZE );

      return;
   }

   /* if command comes from listmethods */

   else if ( strncmp( buf, "lme", 3 ) == 0 )
   {
      /* discard the first two tokens */
      strtok2( &pointer, &delimiter, " " );
      strtok2( &pointer, &delimiter, " " );

      jvm = atoi( strtok2( &pointer, &delimiter, " " ) );

      if ( ( jvm < 0 ) || ( jvm >= *server_index ) )
      {
         sprintf( buf, "-10000" );
         ( void ) write( slave_socket, largebuf, BUFSIZE );
         return;
      }

      sprintf( largebuf, "%d", JVM_servers[ jvm ].port_number );
      s = connectTCP( JVM_servers[ jvm ].host, largebuf );
      ( void ) write( s, buf, LINELEN );

      for ( cc = 0; cc < BUFSIZE; cc += n )
      {
         n = read( s, largebuf, BUFSIZE );
         if ( n < 0 )
            errexit( "read: %s\n", strerror( errno ) );
      }

      ( void ) write( slave_socket, largebuf, BUFSIZE );

      return;
   }

   /* if command comes from instantiate */

   else if ( strncmp( buf, "ins", 3 ) == 0 )
   {
      /* discard the first token */
      strtok2( &pointer, &delimiter, " " );

      jvm = atoi( strtok2( &pointer, &delimiter, " " ) );

      if ( ( jvm < 0 ) || ( jvm >= *server_index ) )
      {
         sprintf( largebuf, "-10000" );
         ( void ) write( slave_socket, largebuf, BUFSIZE );
         return;
      }

      sprintf( largebuf, "%d", JVM_servers[ jvm ].port_number );
      s = connectTCP( JVM_servers[ jvm ].host, largebuf );
      ( void ) write( s, buf, LINELEN );

      for ( cc = 0; cc < LINELEN; cc += n )
      {
         n = read( s, buf, LINELEN );
         if ( n < 0 )
            errexit( "read: %s\n", strerror( errno ) );
      }

      ( void ) write( slave_socket, buf, LINELEN );

      if ( atoi( buf ) == -20000 )
         return;

      if ( atoi( buf ) == -30000 )
         return;

      for ( cc = 0; cc < BUFSIZE; cc += n )
      {
         n = read( slave_socket, largebuf, BUFSIZE );
         if ( n < 0 )
            errexit( "read: %s\n", strerror( errno ) );
      }

      ( void ) write( s, largebuf, BUFSIZE );

      for ( cc = 0; cc < LINELEN; cc += n )
      {
         n = read( s, buf, LINELEN );
         if ( n < 0 )
            errexit( "read: %s\n", strerror( errno ) );
      }

      ( void ) write( slave_socket, buf, LINELEN );

      return;
   }

   /* if command comes from callmethod */

   else if ( strncmp( buf, "clm", 3 ) == 0 )
   {
      /* discard the first token */
      strtok2( &pointer, &delimiter, " " );

      jvm = atoi( strtok2( &pointer, &delimiter, " " ) );

      if ( ( jvm < 0 ) || ( jvm >= *server_index ) )
      {
         sprintf( largebuf, "-10000" );
         ( void ) write( slave_socket, largebuf, BUFSIZE );
         return;
      }

      sprintf( largebuf, "%d", JVM_servers[ jvm ].port_number );
      s = connectTCP( JVM_servers[ jvm ].host, largebuf );
      ( void ) write( s, buf, LINELEN );

      for ( cc = 0; cc < LINELEN; cc += n )
      {
         n = read( s, buf, LINELEN );
         if ( n < 0 )
            errexit( "read: %s\n", strerror( errno ) );
      }

      ( void ) write( slave_socket, buf, LINELEN );

      if ( atoi( buf ) == -20000 )
         return;

      if ( atoi( buf ) == -30000 )
         return;

      if ( atoi( buf ) == -35000 )
         return;

      for ( cc = 0; cc < BUFSIZE; cc += n )
      {
         n = read( slave_socket, largebuf, BUFSIZE );
         if ( n < 0 )
            errexit( "read: %s\n", strerror( errno ) );
      }

      ( void ) write( s, largebuf, BUFSIZE );

      for ( cc = 0; cc < LINELEN; cc += n )
      {
         n = read( s, buf, LINELEN );
         if ( n < 0 )
            errexit( "read: %s\n", strerror( errno ) );
      }

      ( void ) write( slave_socket, buf, LINELEN );

      return;
   }
}
