// $Id: pjvm.java,v 1.6 2001/12/03 17:15:00 eric Exp $

/* SUMMARY
 *
 * pjvm.java - pjvm.class
 *
 * pjvm.class provides the Java side of the Reflection utilities needed
 * for PJVM to function.  Although most of the work of pjvm.class could
 * be done on the native side in C, many of these functions are more
 * easily called from Java, and using these wrappers helps to speed
 * execution and make the native code more readable.
 *
 *
 * REVISION HISTORY
 *
 * $Log: pjvm.java,v $
 * Revision 1.6  2001/12/03 17:15:00  eric
 * Removed overloaded loadObject().
 * Added collectGarbage() method.
 *
 * Revision 1.5  2001/11/27 22:11:34  eric
 * Removed getArguments()
 *
 * Revision 1.4  2001/11/14 18:12:17  eric
 * Modified to allow user to set garbage collector flag when instantiating an object, which will signal the GC to de-reference the object in question when called.
 *
 * Revision 1.3  2001/11/07 02:42:58  eric
 * Consolidated getParameters()
 *
 * Revision 1.2  2001/11/04 04:05:02  eric
 * Corrected faulty conversion of remote filename into URL
 *
 * Revision 1.1  2001/10/23 21:30:02  eric
 * Initial revision
 *
 */

import java.io.*;
import java.util.*;
import java.net.*;
import java.lang.reflect.*;
import javax.swing.*;


//  Class pjvm
//  --------------------------------------------------------------------
/**
  *     Class pjvm represents an interface to a persistent JVM.
  *     Instead of loading a new instance of the VM each time that user
  *     wants to load or instantiate a new application, the persistent
  *     Java Virtual Machine will allow users to leave all loaded
  *     and instantiated classes in memory, and manipulate those
  *     classes and objects dynamically.
  *
  *     The JVM itself must be instantiated elsewhere; however, the
  *     pjvm class behaves as a front-end for the Java Reflection
  *     mechanism, facilitating the manipulation of objects already
  *     in the system.
  *
  *     @author Eric J. Shamow
  *     @version $Id: pjvm.java,v 1.6 2001/12/03 17:15:00 eric Exp $
  */
public class pjvm {
   private Vector loadedObjects = new Vector();
   private Vector loadedClasses = new Vector();


   //  Constructor pjvm()
   //  -----------------------------------------------------------------
   /**
     *   Constructor for a new pjvm.  Each additional user will
     *   get a new instance of pjvm to go with his/her own JVM instance.
     *
     */

   public pjvm() {}
    

   
   //  Method getObjectVector()
   //  -----------------------------------------------------------------
   /**
     *   This method returns a reference to the <code>loadedObjects
     *   </code> Vector.
     *
     *     @return A vector with references to all loaded objects.
     */
   public Vector getObjectVector() {
      return loadedObjects;
   }

   //  Method getClassVector()
   //  -----------------------------------------------------------------
   /**
     *   This method returns a reference to the <code>loadedClasses
     *   </code> Vector.
     *
     *     @return A vector with references to all loaded classes.
     */
   public Vector getClassVector() {
      return loadedClasses;
   }

   //  Method loadClass()
   //  -----------------------------------------------------------------
   /**
     *   This method attempts to load the class with name <code>
     *   currentClass</code>.  <code>currentClass</code> should be a
     *   pathname and filename, with or without extension.
     *   If the class is successfully located,
     *   loadClass() will load the class and add it to an internal
     *   record of loaded classes, awaiting manipulation.
     *
     *     @param  currentClass The formatted name of the class to load.
     *     @return True if the class loaded, false otherwise.
     */
   public boolean loadClass( String currentClass, boolean classLocal ) {
      File currentClassFile;
      URL currentClassURL, serverURLs[] = new URL[ 1 ];
      URLClassLoader cl;
      String classname, URLpathname;
      int replacementIndex = 0;
      boolean loadSuccess = false;

      // if currentClass doesn't end in ".class," add it

      if ( currentClass.charAt( currentClass.length() - 6 ) != '.' )
         currentClass = currentClass + ".class";

      currentClassFile = new File( currentClass );

      // if class is local
      if ( classLocal == true ) {
         if ( !currentClassFile.exists() ) {
            System.err.println( "File doesn't exist." );
            return false; 
         }

        // create the URL

         try {
            URLpathname = currentClassFile.getCanonicalPath();

            // remove the filename from the URL

            replacementIndex = URLpathname.length();
            for ( int i = 0; i < URLpathname.length(); i++ ) {
               if ( URLpathname.charAt( i ) == '/' ) {
                  replacementIndex = i;
               }
            }

            URLpathname = URLpathname.substring( 0, 
                                                 replacementIndex + 1 );
            currentClassURL = ( new File( URLpathname ) ).toURL();

            // translate that URL into an array for the URLClassLoader

            serverURLs[ 0 ] = currentClassURL;
         } catch( Exception e ) { 
            System.err.println( e ); 
            return false;
         }

      }

      // otherwise class is remote
      else {
         // create the URL

         try {
         URLpathname = currentClass;
         replacementIndex = URLpathname.length();
         for ( int i = 0; i < URLpathname.length(); i++ ) {
            if ( URLpathname.charAt( i ) == '/' ) {
               replacementIndex = i;
            }
         }

         URLpathname = URLpathname.substring( 0, replacementIndex + 1 );
         currentClassURL = new URL( URLpathname );

            // translate that URL into an array for the URLClassLoader

            serverURLs[ 0 ] = currentClassURL;
         } catch( Exception e ) { 
            System.err.println( e );
            return false;
         }
      }

      // get the actual filename from currentClass

      classname = currentClassFile.getName();
      
      for ( int i = 0; i < classname.length(); i++ ) {
         if ( classname.charAt( i ) == '.' ) {
            if ( classname.substring( i + 1, classname.length() ).
                                      compareTo( "class" ) == 0 )
              classname = classname.substring( 0, i );
         }
      }

      // try to instantiate classloader using new URL

      try {
         cl = new URLClassLoader( serverURLs );
         Class temp = Class.forName( classname, true, cl );
         loadedClasses.add( temp );
      } catch( Exception e ) { 
         System.err.println( e );
         return false;
      }

      return true;
   }


   //  Method getConstructors()
   //  -----------------------------------------------------------------
   /**
     *   Takes a Class as an argument, and returns a vector of
     *   Constructors which can be used to create an instance of that
     *   class.
     *
     *     @param  selectedClass Class to scan for constructors.
     *     @return A vector of constructors for this class.
     */
   public Vector getConstructors( Class selectedClass ) {
      Vector declaredConstructors = new Vector();
      Constructor[] constructorArray = selectedClass.
                                       getDeclaredConstructors();

      for ( int i = 0; i < constructorArray.length; i++ ) {
         declaredConstructors.add( constructorArray[ i ] );
      }

      return declaredConstructors;
   }


   //  Method getMethods()
   //  -----------------------------------------------------------------
   /**
     *   Takes a Class as an argument, and returns a vector of
     *   Methods accessible from the class.
     *
     *     @param  selectedClass Class to scan for constructors.
     *     @return A vector of methods within this class.
     *
     */
   public Vector getMethods( Class selectedClass ) {
      Vector methods = new Vector();
      Method[] methodArray = selectedClass.getMethods();

      for ( int i = 0; i < methodArray.length; i++ ) {
         methods.add( methodArray[ i ] );
      }

      return methods;
   }


   //  Method getDeclaredMethods()
   //  -----------------------------------------------------------------
   /**
     *   Takes a Class as an argument, and returns a vector of
     *   Methods accessible from the class, that have been declared
     *   explicitly by the class.
     *
     *     @param  selectedClass Class to scan for constructors.
     *     @return A vector of declared methods within this class.
     *
     */
   public Vector getDeclaredMethods( Class selectedClass ) {
      Vector methods = new Vector();
      Method[] methodArray = selectedClass.getDeclaredMethods();

      for ( int i = 0; i < methodArray.length; i++ ) {
         methods.add( methodArray[ i ] );
      }

      return methods;
   }


   //  Method getParameters()
   //  -----------------------------------------------------------------
   /**
     *   Takes a Constructor as an argument, and extracts 
     *   parameter types.
     *
     *   @param  selectedConstructor The selected constructor for the
     *                               class.
     *   @return Parameter types for the constructor.
     */
   public Vector getParameters( 
                 Constructor selectedConstructor ) {
      Vector constructorParameters = new Vector();
      Class[] parameterArray = selectedConstructor.
                               getParameterTypes();
      
      for ( int i = 0; i < parameterArray.length; i++ ) {
         constructorParameters.add( parameterArray[ i ] );
      }

      return constructorParameters;
   }


   //  Method getParameters()
   //  -----------------------------------------------------------------
   /**
     *   Takes a Method as an argument, and extracts parameter types.
     *
     *   @param  selectedMethod The selected method for the class.
     *   @return Parameter types for the method.
     */
   public Vector getParameters( Method selectedMethod ) {
      Vector methodParameters = new Vector();
      Class[] parameterArray = selectedMethod.getParameterTypes();

      for ( int i = 0; i < parameterArray.length; i++ ) {
         methodParameters.add( parameterArray[ i ] );
      }
      
      return methodParameters;
   }


   //  Method loadObject()
   //  -----------------------------------------------------------------
   /**
     *   This method attempts to instantiate the object contained in
     *   <code>loadedClass</code>.  It returns true if the object
     *   is loaded correctly.
     *
     *     @param  selectedConstructor  The constructor to use in
     *                                  instantiation.
     *     @param  argumentVector       Arguments to pass to the
     *                                  constructor.
     *     @param  garbageCollectorFlag Flag to indicate that this
     *                                  object will be de-referenced
     *                                  when the garbage collector is
     *                                  called.
     *     @return True if the object was instantiated. 
     */
   public boolean loadObject( Constructor selectedConstructor,
                              Vector argumentVector,
                              boolean garbageCollectorFlag ) {
      Object loadedObject = null,
             arguments[] = argumentVector.toArray();

      try {
         loadedObject = selectedConstructor.newInstance( arguments );
      }
      catch( Exception e ) {
         System.err.println( e );
         return false;
      }

      return storeObject( loadedObject, garbageCollectorFlag );
   }


   //  Method storeObject()
   //  -----------------------------------------------------------------
   /**
     *   This method stores the object specified in the internal Vector
     *   of instantiated objects.
     *
     *     @param object               The object to store.
     *     @param garbageCollectorFlag Flag to indicate that this object
     *                                 will be de-referenced when the
     *                                 garbage collector is called.
     */

   public boolean storeObject( Object object, 
                               boolean garbageCollectorFlag ) {
      Object vectorEntry[] = new Object[ 2 ];

      if ( object != null )
      {
         vectorEntry[ 0 ] = object;
         vectorEntry[ 1 ] = new Boolean( garbageCollectorFlag );
         loadedObjects.add( vectorEntry );
         return true;
      }
      else
         return false;
   }

   
   //  Method collectGarbage()
   //  -----------------------------------------------------------------
   /**
     *   This method clears all objects marked for garbage collection
     *   from the object vector.
     *
     */

   public void collectGarbage() {
      boolean reachedEnd = false;
      int currentIndex = 0;
      Object vectorEntry[];

      while ( reachedEnd == false ) {
         if ( currentIndex >= loadedObjects.size() )
            reachedEnd = true;
         else {
            vectorEntry = ( Object[] ) loadedObjects.elementAt( 
                                                     currentIndex );

            if ( ( ( Boolean ) vectorEntry[ 1 ] ).booleanValue() == 
                                                  true )
               loadedObjects.removeElementAt( currentIndex );
            else
               currentIndex++;

         }
      }
   }

   //  Method executeMethod()
   //  -----------------------------------------------------------------
   /**
     *   This method attempts to execute the method specified by
     *   <code>meth</code>, on the object specified by <code>target
     *   </code>. It returns an Object representing the return
     *   value of the method. If nothing is returned, it passes null.
     *
     *     @param  meth   A reference to the method to execute.
     *     @param  args   Arguments to pass to the method.
     *     @param  target An object to operate upon, null if static.
     *     @return Object A reference to the return value of the method.
     */
   public Object executeMethod( Method meth, Vector args,
                                             Object target ) {
      Object arguments[];

      if ( args.size() != 0 ) {
         arguments = args.toArray();
      } else {
         arguments = null;
      }

      try {

         // if the method is static, it doesn't need to operate an a
         // loaded object

         if ( Modifier.isStatic( meth.getModifiers() ) ) {
            return meth.invoke( null, arguments );
         }

         // otherwise, the user needs to specify an object in memory.

         else {
System.out.println( "The target is " + target );
            return meth.invoke( target, arguments );
         }

      // If a dynamically-executed method throws an exception, that
      // exception is wrapped in an InvocationTargetException before
      // it's passed to the referencing class.  We want the user to
      // know what the problem is with the original code, so we need
      // to extract the target exception.

      } catch( InvocationTargetException ite ) {
         System.err.println( "\nYour application threw an exception:\n" 
                           );
         ite.getTargetException().printStackTrace();
      } catch( Exception re ) { System.err.println( re ); }

      return null;
   }

}

