/* $Id: pjvms.c,v 1.8 2001/12/06 16:08:04 eric Exp $ */

/* SUMMARY
 *
 * pjvms.c - pjvms
 *
 * The program pjvms acts as a server for PJVM services.  The server 
 * runs as a background or daemon process, and accepts
 * connections on the port provided as an argument.  With each request 
 * for a new JVM, PJVMS launches a subprocess with a new VM and provides
 * a port number through which that JVM can be accessed.  PJVMS keeps 
 * track of all launched VMs.
 *
 *
 * REVISION HISTORY
 *
 * $Log: pjvms.c,v $
 * Revision 1.8  2001/12/06 16:08:04  eric
 * Removed extraneous debugging messages.
 *
 * Revision 1.7  2001/12/03 17:16:30  eric
 * Added support for instantiation finalizer.
 * Added support for objects to instantiate.
 * Added support for command-line nesting of object instantiation.
 * Corrected erroneous error reporting in instantiate.
 * Fixed bug in listobjects which was causing classes to display instead of
 * objects.
 * Fixed bug in listobjects, allowing full-length object descriptions to
 * appear.
 *
 * Revision 1.6  2001/11/27 22:08:59  eric
 * Added support for listobjects
 *
 * Revision 1.5  2001/11/19 17:10:00  eric
 * Added further error/debugging output on failure to instantiate object.
 * Fixed inaccurate detection of parameters in instantiate.
 * Added support for strings on command line in instantiate
 *
 * Revision 1.4  2001/11/14 18:14:20  eric
 * Corrected problems with local classloading and added rudimentary support for instantiate(1)
 *
 * Revision 1.3  2001/11/04 05:29:15  eric
 * Added listmethods
 *
 * Revision 1.2  2001/10/31 00:28:52  eric
 * Completed listconstructors
 *
 * Revision 1.1  2001/10/23 21:30:02  eric
 * Initial revision
 *
 */

#define _USE_BSD
#include <sys/types.h>
#include <sys/signal.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/errno.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/sem.h>
#include <netinet/in.h>
#include <netdb.h>

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

#include <jni.h>

#include "reaper.h"
#include "strtok2.h"
#include "errexit.h"
#include "passiveTCP.h"

#define QLEN    32            /* maximum connection queue length */
#define BUFSIZE 4096
#define LINELEN 128
#define REMOTE 0              /* constants for loadclass source */
#define LOCAL 1
#define UNDEFINED 5
#define TRUE 1
#define FALSE 0

#define jvector jobject
#define jconstructor jobject
#define jmethod jobject
#define jparameter jobject

extern int errno;

int pjvms( int file_descriptor );


/*  Function main()
 *  --------------------------------------------------------------------
 *
 *    Concurrent TCP server for PJVM service.
 *
 */

int main( int argc, char *argv[] )
{
   char *service = "pjvms";
   struct sockaddr_in client_address;
   unsigned int address_length;
   int master_socket, slave_socket;

   switch ( argc )
   {
      case 1:  break;
      case 2:  service = argv[ 1 ];
               break;
      default: errexit( "usage: pjvms [port]\n" );
   }

   /* create a passive socket to accept connections from clients */
   master_socket = passiveTCP( service, QLEN );

   /* prevent zombie processes */
   ( void ) signal( SIGCHLD, reaper ); 

   for ( ;; )
   {
      /* open a new socket to deal with the incoming
       * connection */
      address_length = sizeof( client_address );
      slave_socket = accept( master_socket, ( struct sockaddr * )
                             &client_address, &address_length );

      if ( slave_socket < 0 )
      {
         if ( errno == EINTR )
	    continue;
         errexit( "accept: %s\n", strerror( errno ) );
      }

      switch ( fork() )
      {
         case  0: /* child */
	               ( void ) close( master_socket );
                  exit( pjvms( slave_socket ) );
	               /* FALL-THROUGH */
         default: /* parent */
	               ( void ) close( slave_socket );
		            break;
         case -1: errexit( "fork: %s\n", strerror( errno ) );
      }
   }
}


/*  Function TCPpjvms()
 *  --------------------------------------------------------------------
 *
 *    Provide JVMs and PJVM reflection utilities to clients.
 *
 */

int pjvms( int fd )
{
   char buf[ LINELEN + 1 ], outbuf[ LINELEN + 1 ], *pointer = buf,
        delimiter, *object_chars, largebuf[ BUFSIZE + 1 ],
        largebuf2[ BUFSIZE + 1 ];
   int master_socket, slave_socket, cc, n, portnum, bound, i, argc,
       vector_size, vector_size_2, 
       remaining_space, selected_class_number,
       declared_only = FALSE, selected_constructor_number,
       selected_method_number, selected_object_number,
       arg_count = 0, call_finalizer = FALSE;
   unsigned int address_length;
   struct sockaddr_in client_address;
   long result;
   JavaVM *jvm;
   JNIEnv *env;
   jclass pjvm_cls, vector_class, object_class, selected_class;
   jobject pjvm;
   jstring class_name_string, object_string;
   jmethodID load_class_ID, pjvm_init_mid, get_class_vector_ID,
             vector_size_ID, vector_element_at_ID, object_tostring_ID,
             get_constructors_ID, get_methods_ID, get_parameters_ID,
             vector_constructor_ID, load_object_ID,
             get_object_vector_ID, collect_garbage_ID,
             execute_method_ID, store_object_ID;
   jboolean class_loaded, class_local, instantiate_failed = JNI_FALSE,
            garbage_collected, static_method = JNI_FALSE,
            declared_method = JNI_FALSE;
   jthrowable exception;
   JavaVMOption *options;
   JavaVMInitArgs vm_args;
   jvector class_vector, class_constructor_vector, class_method_vector,
           constructor_parameter_vector, argument_vector, object_vector,
           method_vector, method_parameter_vector;
   jconstructor class_constructor;
   jmethod class_method, selected_method;
   jobjectArray selected_object_array;
   jobject selected_object, loaded_object;


   for ( cc = 0; cc < LINELEN; cc += n )
   {
      n = read( fd, buf, LINELEN );
      if ( n < 0 )
         errexit( "echo read: %s\n", strerror( errno ) );
   }

   /* if request is coming from requestjvm, we need to fork, create
    * a new JVM, and pass back a port number through which the JVM
    * can be accessed */
   
   if ( strncmp( buf, "jvm", 3 ) == 0 )
   {
      /* fork eliminated here - replace if necessary */

      /* advance past "jvm" in the buffer */
      pointer += 4;

      while ( 1 )
      {
         argc = atoi( strtok2( &pointer, &delimiter, " \0" ) );

         /* if there are arguments, set up JavaVMOptions to pass them to
            the JVM */
         if ( argc > 0 )
         {
            options = ( JavaVMOption * ) malloc( sizeof( JavaVMOption ) 
                                         * ( argc + 1 ) );
            if ( options == NULL )
            {
               n = -1;
               break;
            }

            for ( i = 1; i <= argc; i++ )
               options[ i ].optionString = strtok2( &pointer, 
                                                    &delimiter,
                                                    " " ); 

         }
         else
         {
            options = ( JavaVMOption * ) malloc( sizeof( 
                                                 JavaVMOption ) );
            if ( options == NULL )
            {
               n = -1;
               break;
            }
         }

         options[ 0 ].optionString = "-XX:+AllowUserSignalHandlers";

         vm_args.version = JNI_VERSION_1_2;
         vm_args.options = options;
         vm_args.nOptions = argc + 1;
         vm_args.ignoreUnrecognized = JNI_TRUE;

         result = JNI_CreateJavaVM( &jvm, ( void ** ) &env, &vm_args );

         if ( result == JNI_ERR )
         {
            n = -2;
            break;
         }

         pjvm_cls = ( *env )->FindClass( env, "pjvm" );
         if ( pjvm_cls == NULL )
         {
            exception = ( *env )->ExceptionOccurred( env );
            if ( exception )
               ( *env )->ExceptionClear( env );
            n = -3;
            break;
         }

         pjvm_init_mid = ( *env )->GetMethodID( env, pjvm_cls, "<init>",
                                                 "()V" );
         if ( pjvm_init_mid == NULL )
         {
            exception = ( *env )->ExceptionOccurred( env );
            if ( exception )
               ( *env )->ExceptionClear( env );
            n = -4;
            break;
         }

         pjvm = ( *env )->NewObject( env, pjvm_cls, pjvm_init_mid );
         if ( pjvm == NULL )
         {
            exception = ( *env )->ExceptionOccurred( env );
            if ( exception )
               ( *env )->ExceptionClear( env );
            n = -5;
            break;
         }

         n = 0;
         break;
      }
   }

   sprintf( outbuf, "%d", n );
   if ( write( fd, outbuf, LINELEN + 1 ) < 0 )
      errexit( "write: %s\n", strerror( errno ) );

   /* creates a passive socket to accept connection from clients -
      chooses a random port number and attempts to bind */

   for ( srand( ( unsigned int ) time( NULL ) ), bound = 0; !bound; )
   {
      portnum = rand() % 65536;
      sprintf( buf, "%d", portnum );

      if ( ( master_socket = passiveTCP( buf, QLEN ) ) < 0 )
         continue;

      else
         bound = 1;
   }

   sprintf( buf, "%d", portnum );
   ( void ) write( fd, buf, LINELEN ); 

   /* accept connections from clients wishing to communicate directly
      with the JVM. */

   for ( ;; )
   {
      pointer = buf;
      address_length = sizeof( client_address );
      slave_socket = accept( master_socket, ( struct sockaddr * )
                             &client_address, &address_length );
      
      if ( slave_socket < 0 )
         errexit( "accept failed: %s\n", strerror( errno ) );
      
      for ( cc = 0; cc < LINELEN; cc += n )
      {
         n = read( slave_socket, buf, LINELEN );
         if ( n < 0 )
            errexit( "read: %s\n", strerror( errno ) );
      }

      /* if command comes from killjvm */

      if ( strncmp( buf, "kvm", 3 ) == 0 )
      {
         /* The below is commented out because, as of JDK 1.3, Sun has
            deactivated DestroyJavaVM().  It should be re-enabled
            when DestroyJavaVM() is properly implemented.
         sprintf( buf, "%d", ( int ) ( *jvm )->DestroyJavaVM( jvm ) );
         */
         sprintf( buf, "0" );
         ( void ) write( slave_socket, buf, LINELEN );

         ( void ) close( slave_socket );
         return atoi( buf );
      }

      /* if command comes from loadclass */

      else if ( strncmp( buf, "ldc", 3 ) == 0 )
      {
         /* discard first 2 tokens of buffer */

         strtok2( &pointer, &delimiter, " " );
         strtok2( &pointer, &delimiter, " " );

         /* if class file is from a remote location */

         if ( atoi( strtok2( &pointer, &delimiter, " " ) ) == REMOTE )
            class_local = JNI_FALSE;
         else
            class_local = JNI_TRUE;

         /* convert class name into JString */

         class_name_string = ( *env )->NewStringUTF( env, pointer );
         if ( class_name_string == NULL )
            errexit( "Could not initialize class string.\n" );

         /* get pjvm.loadClass() */

         load_class_ID = ( *env )->GetMethodID( env, pjvm_cls,
                                   "loadClass",
                                   "(Ljava/lang/String;Z)Z" );
         if ( load_class_ID == NULL )
            errexit( "Could not find pjvm.loadClass().\n" );

         /* try to load class */

         class_loaded = ( *env )->CallBooleanMethod( env, pjvm,
                                  load_class_ID, class_name_string,
                                  class_local );

         /* if an exception was thrown in URLClassLoader, handle */

         exception = ( *env )->ExceptionOccurred( env );
         if ( exception )
         {
            ( *env )->ExceptionDescribe( env );
            ( *env )->ExceptionClear( env );
            sprintf( buf, "-1" );
            ( void ) write( slave_socket, buf, LINELEN );

            ( void ) close( slave_socket );
            continue;
         }

         if ( class_loaded == JNI_FALSE )
         {
            sprintf( buf, "-1" );
            ( void ) write( slave_socket, buf, LINELEN );

            ( void ) close( slave_socket );
            continue;
         }

         sprintf( buf, "1" );
         ( void ) write( slave_socket, buf, LINELEN );

         ( void ) close( slave_socket );
         continue;
      }

      /* if command comes from listclass */

      else if ( strncmp( buf, "lsc", 3 ) == 0 )
      {
         /* Get getClassVector() method */

         get_class_vector_ID = ( *env )->GetMethodID( env, pjvm_cls,
                               "getClassVector", 
                               "()Ljava/util/Vector;" );

         if ( get_class_vector_ID == NULL )
            errexit( "Can't find pjvm.getClassVector().\n" );

         /* get vector of classes */

         class_vector = ( *env )->CallObjectMethod( env, pjvm,
                                                 get_class_vector_ID );

         if ( class_vector == NULL )
            errexit( "Could not retrieve record of loaded classes.\n" );

         /* get Vector class */

         vector_class = ( *env )->FindClass( env, "java/util/Vector" );
         if ( vector_class == NULL )
            errexit( "Could not find class Vector.\n" );

         /* get Vector.size() */

         vector_size_ID = ( *env )->GetMethodID( env, vector_class, 
                                                 "size", "()I" );
         if ( vector_size_ID == NULL )
            errexit( "Could not get Vector.size().\n" );
      
         /* get size of class vector */

         vector_size = ( int ) ( *env )->CallIntMethod( env,
                                                       class_vector,
                                                       vector_size_ID );

         /* get Vector.elementAt() */

         vector_element_at_ID = ( *env )->GetMethodID( env, 
                                          vector_class,
                                          "elementAt",
                                          "(I)Ljava/lang/Object;" );
         if ( vector_element_at_ID == NULL )
            errexit( "Could not get Vector.elementAt().\n" );

         /* get Object class */

         object_class = ( *env )->FindClass( env, "java/lang/Object" );
         if ( object_class == NULL )
            errexit( "Could not find class Object.\n" );

         /* get Object.toString() */

         object_tostring_ID = ( *env )->GetMethodID( env, object_class,
                                               "toString",
                                               "()Ljava/lang/String;" );
         if ( object_tostring_ID == NULL )
            errexit( "Could not get Object.toString().\n" );
         /* generate list of loaded classes */

         remaining_space = BUFSIZE;

         strcpy( largebuf, "" );   /* clear buffer */
         
         for ( i = 0; i < vector_size; i++ )
         {
            selected_class = ( *env )->CallObjectMethod( env, 
                                    class_vector,
                                    vector_element_at_ID, ( jint ) i );
            if ( selected_class == NULL )
               errexit( "Error reading class vector.\n" );
         
            object_string = ( *env )->CallObjectMethod( env,
                                                     selected_class,
                                                     object_tostring_ID
                                                      );
            if ( object_string == NULL )
               errexit( "Could not convert Class to String.\n" );

            object_chars = ( char * ) ( *env )->GetStringUTFChars( env,
                                                object_string, NULL );
            if ( object_chars == NULL )
               errexit( "Failed to convert String to char array.\n" );

            snprintf( buf, LINELEN, "%d %s\n", i, object_chars );

            strncat( largebuf, buf, remaining_space );
            remaining_space -= strlen( buf );

            /* free character string */

            ( *env )->ReleaseStringUTFChars( env, object_string,
                                             object_chars );
         }

         ( void ) write( slave_socket, largebuf, BUFSIZE );
         continue;
      }

      /* if command comes from listobjects */

      else if ( strncmp( buf, "lso", 3 ) == 0 )
      {
         /* Get getObjectVector() method */

         get_object_vector_ID = ( *env )->GetMethodID( env, pjvm_cls,
                               "getObjectVector", 
                               "()Ljava/util/Vector;" );

         if ( get_class_vector_ID == NULL )
            errexit( "Can't find pjvm.getObjectVector().\n" );

         /* get vector of objects */

         object_vector = ( *env )->CallObjectMethod( env, pjvm,
                                               get_object_vector_ID );

         if ( class_vector == NULL )
            errexit( "Could not retrieve record of instantiated "
                     "objects.\n" );

         /* get Vector class */

         vector_class = ( *env )->FindClass( env, "java/util/Vector" );
         if ( vector_class == NULL )
            errexit( "Could not find class Vector.\n" );

         /* get Vector.size() */

         vector_size_ID = ( *env )->GetMethodID( env, vector_class, 
                                                 "size", "()I" );
         if ( vector_size_ID == NULL )
            errexit( "Could not get Vector.size().\n" );
      
         /* get size of object vector */

         vector_size = ( int ) ( *env )->CallIntMethod( env,
                                                       object_vector,
                                                       vector_size_ID );

         /* get Vector.elementAt() */

         vector_element_at_ID = ( *env )->GetMethodID( env, 
                                          vector_class,
                                          "elementAt",
                                          "(I)Ljava/lang/Object;" );
         if ( vector_element_at_ID == NULL )
            errexit( "Could not get Vector.elementAt().\n" );

         /* get Object class */

         object_class = ( *env )->FindClass( env, "java/lang/Object" );
         if ( object_class == NULL )
            errexit( "Could not find class Object.\n" );

         /* get Object.toString() */

         object_tostring_ID = ( *env )->GetMethodID( env, object_class,
                                               "toString",
                                               "()Ljava/lang/String;" );
         if ( object_tostring_ID == NULL )
            errexit( "Could not get Object.toString().\n" );

         /* generate list of loaded objects */

         remaining_space = BUFSIZE;

         strcpy( largebuf, "" );   /* clear buffer */
         
         for ( i = 0; i < vector_size; i++ )
         {
            selected_object_array = ( *env )->CallObjectMethod( env, 
                                    object_vector,
                                    vector_element_at_ID, ( jint ) i );
            if ( selected_class == NULL )
               errexit( "Error reading object vector.\n" );
         
            selected_object = ( *env )->GetObjectArrayElement( env,
                                        selected_object_array, 0 );
            if ( selected_object == NULL )
               errexit( "Error reading object array.\n" );

            object_string = ( *env )->CallObjectMethod( env,
                                                     selected_object,
                                                     object_tostring_ID
                                                      );
            if ( object_string == NULL )
               errexit( "Could not convert Class to String.\n" );

            object_chars = ( char * ) ( *env )->GetStringUTFChars( env,
                                                object_string, NULL );
            if ( object_chars == NULL )
               errexit( "Failed to convert String to char array.\n" );

            snprintf( largebuf2, BUFSIZE, "%d %s\n", i, object_chars );

            strncat( largebuf, largebuf2, remaining_space );
            remaining_space -= strlen( largebuf2 );

            /* free character string */

            ( *env )->ReleaseStringUTFChars( env, object_string,
                                             object_chars );
         }

         ( void ) write( slave_socket, largebuf, BUFSIZE );
         continue;
      }


      /* if command comes from listconstructors */

      else if ( strncmp( buf, "lco", 3 ) == 0 )
      {
         /* clear largebuf */

         strcpy( largebuf, "" );

         /* discard the first two tokens */
         strtok2( &pointer, &delimiter, " " );
         strtok2( &pointer, &delimiter, " " );

         /* Get getClassVector() method */

         get_class_vector_ID = ( *env )->GetMethodID( env, pjvm_cls,
                               "getClassVector", 
                               "()Ljava/util/Vector;" );
         if ( get_class_vector_ID == NULL )
            errexit( "Can't find pjvm.getClassVector().\n" );

         /* get vector of classes */

         class_vector = ( *env )->CallObjectMethod( env, pjvm,
                                                 get_class_vector_ID );
         if ( class_vector == NULL )
            errexit( "Could not retrieve record of loaded classes.\n" );

         /* get Vector class */

         vector_class = ( *env )->FindClass( env, "java/util/Vector" );
         if ( vector_class == NULL )
            errexit( "Could not find class Vector.\n" );

         /* get Vector.size() */

         vector_size_ID = ( *env )->GetMethodID( env, vector_class, 
                                                 "size", "()I" );
         if ( vector_size_ID == NULL )
            errexit( "Could not get Vector.size().\n" );
      
         /* get size of class vector */

         vector_size = ( int ) ( *env )->CallIntMethod( env,
                                                       class_vector,
                                                       vector_size_ID );

         /* get Vector.elementAt() */

         vector_element_at_ID = ( *env )->GetMethodID( env, 
                                          vector_class,
                                          "elementAt",
                                          "(I)Ljava/lang/Object;" );
         if ( vector_element_at_ID == NULL )
            errexit( "Could not get Vector.elementAt().\n" );

         /* get Object class */

         object_class = ( *env )->FindClass( env, "java/lang/Object" );
         if ( object_class == NULL )
            errexit( "Could not find class Object.\n" );

         /* determine desired class */

         selected_class_number = atoi( pointer );
         
         /* get size of class vector */

         vector_size = ( int ) ( *env )->CallIntMethod( env,
                                         class_vector, vector_size_ID );

         if ( selected_class_number > ( vector_size - 1 ) )
         {
            sprintf( largebuf, "-20000" );
            ( void ) write( slave_socket, largebuf, BUFSIZE );
            continue;
         }

         /* get the selected class */

         selected_class = ( *env )->CallObjectMethod( env, class_vector,
                                    vector_element_at_ID, ( jint )
                                    selected_class_number );
         if ( selected_class == NULL )
            errexit( "Could not retrieve selected class.\n" );

         /* get pjvm.getConstructors() */

         get_constructors_ID = ( *env )->GetMethodID( env, pjvm_cls,
                                         "getConstructors",
                                         "(Ljava/lang/Class;)"
                                         "Ljava/util/Vector;" );
         if ( get_constructors_ID == NULL )
            errexit( "Could not get pjvm.getConstructors().\n" );

         /* get class constructors */

         class_constructor_vector = ( *env )->CallObjectMethod( env,
                                              pjvm, get_constructors_ID,
                                              selected_class );
         if ( class_constructor_vector == NULL )
            errexit( "Could not get class constructors.\n" );

         /* get size of constructor vector */

         vector_size = ( int ) ( *env )->CallIntMethod( env,
                                         class_constructor_vector,
                                         vector_size_ID );

         remaining_space = BUFSIZE;

         for ( i = 0; i < vector_size; i++ )
         {
            class_constructor = 
                                    ( *env )->CallObjectMethod( env, 
                                    class_constructor_vector,
                                    vector_element_at_ID, ( jint ) i );
            if ( class_constructor == NULL )
               errexit( "Error reading class vector.\n" );
         
            /* get Object class */

            object_class = ( *env )->FindClass( env, 
                                                "java/lang/Object" );
            if ( object_class == NULL )
               errexit( "Could not find class Object.\n" );

            /* get Object.toString() */

            object_tostring_ID = ( *env )->GetMethodID( env, 
                                               object_class,
                                               "toString",
                                               "()Ljava/lang/String;" );
            if ( object_tostring_ID == NULL )
               errexit( "Could not get Object.toString().\n" );

            object_string = ( *env )->CallObjectMethod( env,
                                          class_constructor,
                                          object_tostring_ID );
            if ( object_string == NULL )
               errexit( "Could not convert Class to String.\n" );

            object_chars = ( char * ) ( *env )->GetStringUTFChars( env,
                                                object_string, NULL );
            if ( object_chars == NULL )
               errexit( "Failed to convert String to char array.\n" );

            snprintf( buf, LINELEN, "%d %s\n", i, object_chars );

            strncat( largebuf, buf, remaining_space );
            remaining_space -= strlen( buf );

            /* free character string */

            ( *env )->ReleaseStringUTFChars( env, object_string,
                                             object_chars );
         }

         ( void ) write( slave_socket, largebuf, BUFSIZE );
         continue;
         
      }

      /* if command comes from listmethods */

      else if ( strncmp( buf, "lme", 3 ) == 0 )
      {
         /* clear largebuf */

         strcpy( largebuf, "" );

         /* discard the first token */
         strtok2( &pointer, &delimiter, " " );

         /* determine if declared or inherited methods are desired */
         declared_only = atoi( strtok2( &pointer, &delimiter, " " ) );

         /* discard the third token */
         strtok2( &pointer, &delimiter, " " );

         /* Get getClassVector() method */

         get_class_vector_ID = ( *env )->GetMethodID( env, pjvm_cls,
                               "getClassVector", 
                               "()Ljava/util/Vector;" );
         if ( get_class_vector_ID == NULL )
            errexit( "Can't find pjvm.getClassVector().\n" );

         /* get vector of classes */

         class_vector = ( *env )->CallObjectMethod( env, pjvm,
                                                 get_class_vector_ID );
         if ( class_vector == NULL )
            errexit( "Could not retrieve record of loaded classes.\n" );

         /* get Vector class */

         vector_class = ( *env )->FindClass( env, "java/util/Vector" );
         if ( vector_class == NULL )
            errexit( "Could not find class Vector.\n" );

         /* get Vector.size() */

         vector_size_ID = ( *env )->GetMethodID( env, vector_class, 
                                                 "size", "()I" );
         if ( vector_size_ID == NULL )
            errexit( "Could not get Vector.size().\n" );
      
         /* get size of class vector */

         vector_size = ( int ) ( *env )->CallIntMethod( env,
                                                       class_vector,
                                                       vector_size_ID );


         /* get Vector.elementAt() */

         vector_element_at_ID = ( *env )->GetMethodID( env, 
                                          vector_class,
                                          "elementAt",
                                          "(I)Ljava/lang/Object;" );
         if ( vector_element_at_ID == NULL )
            errexit( "Could not get Vector.elementAt().\n" );

         /* get Object class */

         object_class = ( *env )->FindClass( env, "java/lang/Object" );
         if ( object_class == NULL )
            errexit( "Could not find class Object.\n" );

         /* determine desired class */

         selected_class_number = atoi( pointer );
         
         /* get size of class vector */

         vector_size = ( int ) ( *env )->CallIntMethod( env,
                                         class_vector, vector_size_ID );

         if ( selected_class_number > ( vector_size - 1 ) )
         {
            sprintf( largebuf, "-20000" );
            ( void ) write( slave_socket, largebuf, BUFSIZE );
            continue;
         }

         /* get the selected class */

         selected_class = ( *env )->CallObjectMethod( env, class_vector,
                                    vector_element_at_ID, ( jint )
                                    selected_class_number );
         if ( selected_class == NULL )
            errexit( "Could not retrieve selected class.\n" );

         /* if declared methods only, then get
            pjvm.getDeclaredMethods() */

         if ( declared_only == TRUE )
         {
            get_methods_ID = ( *env )->GetMethodID( env, pjvm_cls,
                                            "getDeclaredMethods",
                                            "(Ljava/lang/Class;)"
                                            "Ljava/util/Vector;" );
            if ( get_methods_ID == NULL )
               errexit( "Could not get pjvm.getDeclaredMethods().\n" );
         }

         /* otherwise, if all methods, then get pjvm.getMethods() */

         else
         {
            get_methods_ID = ( *env )->GetMethodID( env, pjvm_cls,
                                            "getMethods",
                                            "(Ljava/lang/Class;)"
                                            "Ljava/util/Vector;" );
            if ( get_methods_ID == NULL )
               errexit( "Could not get pjvm.getMethods().\n" );
         }

         /* get methods */

         class_method_vector = ( *env )->CallObjectMethod( env,
                                              pjvm, get_methods_ID,
                                              selected_class );
         if ( class_method_vector == NULL )
            errexit( "Could not get methods.\n" );

         /* get size of method vector */

         vector_size = ( int ) ( *env )->CallIntMethod( env,
                                         class_method_vector,
                                         vector_size_ID );

         remaining_space = BUFSIZE;

         for ( i = 0; i < vector_size; i++ )
         {
            class_method = ( *env )->CallObjectMethod( env, 
                                     class_method_vector,
                                     vector_element_at_ID, ( jint ) i );
            if ( class_method == NULL )
               errexit( "Error reading class vector.\n" );
         
            /* get Object class */

            object_class = ( *env )->FindClass( env, 
                                                "java/lang/Object" );
            if ( object_class == NULL )
               errexit( "Could not find class Object.\n" );

            /* get Object.toString() */

            object_tostring_ID = ( *env )->GetMethodID( env, 
                                               object_class,
                                               "toString",
                                               "()Ljava/lang/String;" );
            if ( object_tostring_ID == NULL )
               errexit( "Could not get Object.toString().\n" );

            object_string = ( *env )->CallObjectMethod( env,
                                                     class_method,
                                                     object_tostring_ID
                                                      );
            if ( object_string == NULL )
               errexit( "Could not convert Method to String.\n" );

            object_chars = ( char * ) ( *env )->GetStringUTFChars( env,
                                                object_string, NULL );
            if ( object_chars == NULL )
               errexit( "Failed to convert String to char array.\n" );

            snprintf( buf, LINELEN, "%d %s\n", i, object_chars );

            strncat( largebuf, buf, remaining_space );
            remaining_space -= strlen( buf );

            /* free character string */

            ( *env )->ReleaseStringUTFChars( env, object_string,
                                             object_chars );
         }

         ( void ) write( slave_socket, largebuf, BUFSIZE );
         continue;
         
      }

      /* if command comes from instantiate */

      else if ( strncmp( buf, "ins", 3 ) == 0 )
      {
         /* clear largebuf */

         strcpy( largebuf, "" );

         /* discard the first two tokens */
         strtok2( &pointer, &delimiter, " " );
         strtok2( &pointer, &delimiter, " " );

         /* get selected class number */

         selected_class_number = atoi( strtok2( &pointer, &delimiter,
                                                " " ) );

         /* get selected constructor number */
         selected_constructor_number = atoi( strtok2( &pointer,
                                       &delimiter, " " ) );

         /* determine if object will be garbage collected */
         if ( atoi( strtok2( &pointer, &delimiter, " " ) ) == TRUE )
            garbage_collected = JNI_TRUE;
         else
            garbage_collected = JNI_FALSE;

         /* determine if this is the final object in a chain */
         call_finalizer = atoi( strtok2( &pointer, &delimiter, " " ) );

         /* Get getClassVector() method */

         get_class_vector_ID = ( *env )->GetMethodID( env, pjvm_cls,
                               "getClassVector", 
                               "()Ljava/util/Vector;" );
         if ( get_class_vector_ID == NULL )
            errexit( "Can't find pjvm.getClassVector().\n" );

         /* get vector of classes */

         class_vector = ( *env )->CallObjectMethod( env, pjvm,
                                                 get_class_vector_ID );
         if ( class_vector == NULL )
            errexit( "Could not retrieve record of loaded classes.\n" );

         /* get Vector class */

         vector_class = ( *env )->FindClass( env, "java/util/Vector" );
         if ( vector_class == NULL )
            errexit( "Could not find class Vector.\n" );

         /* get Vector.size() */

         vector_size_ID = ( *env )->GetMethodID( env, vector_class, 
                                                 "size", "()I" );
         if ( vector_size_ID == NULL )
            errexit( "Could not get Vector.size().\n" );
      
         /* get size of class vector */

         vector_size = ( int ) ( *env )->CallIntMethod( env,
                                                       class_vector,
                                                       vector_size_ID );

         /* get Vector.elementAt() */

         vector_element_at_ID = ( *env )->GetMethodID( env, 
                                          vector_class,
                                          "elementAt",
                                          "(I)Ljava/lang/Object;" );
         if ( vector_element_at_ID == NULL )
            errexit( "Could not get Vector.elementAt().\n" );

         /* get Object class */

         object_class = ( *env )->FindClass( env, "java/lang/Object" );
         if ( object_class == NULL )
            errexit( "Could not find class Object.\n" );

         /* get size of class vector */

         vector_size = ( int ) ( *env )->CallIntMethod( env,
                                         class_vector, vector_size_ID );

         if ( selected_class_number > ( vector_size - 1 ) )
         {
            sprintf( largebuf, "-20000" );
            ( void ) write( slave_socket, largebuf, BUFSIZE );
            continue;
         }

         /* get the selected class */

         selected_class = ( *env )->CallObjectMethod( env, class_vector,
                                    vector_element_at_ID, ( jint )
                                    selected_class_number );
         if ( selected_class == NULL )
            errexit( "Could not retrieve selected class.\n" );

         /* get pjvm.getConstructors() */

         get_constructors_ID = ( *env )->GetMethodID( env, pjvm_cls,
                                         "getConstructors",
                                         "(Ljava/lang/Class;)"
                                         "Ljava/util/Vector;" );
         if ( get_constructors_ID == NULL )
            errexit( "Could not get pjvm.getConstructors().\n" );

         /* get class constructors */

         class_constructor_vector = ( *env )->CallObjectMethod( env,
                                              pjvm, get_constructors_ID,
                                              selected_class );
         if ( class_constructor_vector == NULL )
            errexit( "Could not get class constructors.\n" );

         /* get size of constructor vector */

         vector_size = ( int ) ( *env )->CallIntMethod( env,
                                         class_constructor_vector,
                                         vector_size_ID );

         if ( selected_constructor_number > ( vector_size - 1 ) )
         {
            sprintf( largebuf, "-30000" );
            ( void ) write( slave_socket, largebuf, BUFSIZE );
            continue;
         }

         /* attempt to get selected constructor */
         
         class_constructor = ( *env )->CallObjectMethod( env, 
                                    class_constructor_vector,
                                    vector_element_at_ID, ( jint ) 
                                    selected_constructor_number );
         if ( class_constructor == NULL )
            errexit( "Error reading class vector.\n" );
         

         /* get pjvm.getParameters() for Constructors */

         get_parameters_ID = ( *env )->GetMethodID( env, pjvm_cls,
                                     "getParameters",
                                     "(Ljava/lang/reflect/Constructor;)"
                                     "Ljava/util/Vector;" );
         if ( get_parameters_ID == NULL )
            errexit( "Could not get pjvm.getParameters().\n" );


         /* get constructor parameters */

         constructor_parameter_vector = ( *env )->CallObjectMethod( env,
                                        pjvm, get_parameters_ID,
                                        class_constructor );
         if ( constructor_parameter_vector == NULL )
            errexit( "Could not get constructor parameters.\n" );

         /* get size of parameter vector */

         vector_size = ( int ) ( *env )->CallIntMethod( env,
                                         constructor_parameter_vector,
                                         vector_size_ID );

         sprintf( buf, "0" );
         ( void ) write( slave_socket, buf, LINELEN );


         /* if everything has worked out, get parameters from user */

         for ( cc = 0; cc < BUFSIZE; cc += n )
         {
            n = read( slave_socket, largebuf, BUFSIZE );
            if ( n < 0 )
               errexit( "read: %s\n", strerror( errno ) );
         }

         if ( ( vector_size > 0 ) && ( strcmp( largebuf, "" ) == 0 ) )
         {
            sprintf( largebuf, "-40000" );
            ( void ) write( slave_socket, largebuf, BUFSIZE );
            continue;
         }

         /* get empty Vector constructor */

         vector_constructor_ID = ( *env )->GetMethodID( env,
                                           vector_class, "<init>",
                                           "()V" );
         if ( vector_constructor_ID == NULL )
            errexit( "Could not get constructor for class Vector.\n" );

         /* create argument vector */

         argument_vector = ( *env )->NewObject( env, vector_class,
                                                vector_constructor_ID );
         if ( argument_vector == NULL )
            errexit( "Could not create argument Vector.\n" );

         /* populate argument vector */

         pointer = largebuf;


         for ( i = 0; i < vector_size; i++ )
         {
            jobject next_argument;
            jclass type_class;
            jmethodID type_constructor_ID;
            jstring argument;
            jmethodID vector_add_ID;
            jboolean add_successful;
            char *parameter_type;
            jchar char_argument;

            parameter_type = strtok2( &pointer, &delimiter, " " );

            if ( strcmp( parameter_type, "" ) == 0 )
            {
               if ( i < ( vector_size - 1 ) )
                  sprintf( buf, "-50000" );
               else
                  sprintf( buf, "-220000" );
               ( void ) write( slave_socket, buf, LINELEN );
               instantiate_failed = JNI_TRUE;
               continue;
            }

            /* if parameter is a Byte */

            if ( strcmp( parameter_type, "-b" ) == 0 )
            {
               /* determine success or failure */
               parameter_type = strtok2( &pointer, &delimiter, " " );
               if ( strcmp( parameter_type, "Failed:" ) == 0 )
               {
                  sprintf( buf, "-260000" );
                  ( void ) write( slave_socket, buf, LINELEN );
                  instantiate_failed = JNI_TRUE;
               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  /* capture argument */
                  parameter_type = strtok2( &pointer, &delimiter, " " );

                  /* get class Byte */
         
                  type_class = ( *env )->FindClass( env, 
                                      "java/lang/Byte" );
                  if ( type_class == NULL )
                     errexit( "Could not find class Byte.\n" );

                  /* get String constructor for new Byte */

                  type_constructor_ID = ( *env )->GetMethodID( env,
                                               type_class,
                                               "<init>",
                                               "(Ljava/lang/String;)V"
                                               );
                  if ( type_constructor_ID == NULL )
                     errexit( "Could not get Byte( String s ) "
                              "constructor. \n" );

                  /* get argument and convert into jstring */

                  if ( strcmp( parameter_type, "" ) == 0 )
                  {
                     sprintf( buf, "-60000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                  }

               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  argument = ( *env )->NewStringUTF( env, 
                                                     parameter_type );
                  if ( argument == NULL )
                     errexit( 
                           "Could not convert argument to jstring.\n" );

                  /* create Byte */

                  next_argument = ( *env )->NewObject( env, type_class,
                                                    type_constructor_ID,
                                                    argument );
                  /* if argument conversion failed */
                  exception = ( * env )->ExceptionOccurred( env );
                  if ( exception )
                  {
                     ( *env )->ExceptionDescribe( env );
                     ( *env )->ExceptionClear( env );
                  }
                  if ( next_argument == NULL )
                  {
                     sprintf( buf, "-70000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                     continue;
                  }
              }
            }

            /* if parameter is a Boolean */

            else if ( strcmp( parameter_type, "-z" ) == 0 )
            {
               /* determine success or failure */
               parameter_type = strtok2( &pointer, &delimiter, " " );
               if ( strcmp( parameter_type, "Failed:" ) == 0 )
               {
                  sprintf( buf, "-260000" );
                  ( void ) write( slave_socket, buf, LINELEN );
                  instantiate_failed = JNI_TRUE;
               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  /* capture argument */
                  parameter_type = strtok2( &pointer, &delimiter, " " );

                  /* get class Boolean */
         
                  type_class = ( *env )->FindClass( env, 
                                         "java/lang/Boolean" );
                  if ( type_class == NULL )
                     errexit( "Could not find class Boolean.\n" );

                  /* get String constructor for new Boolean */

                  type_constructor_ID = ( *env )->GetMethodID( env,
                                                  type_class,
                                                  "<init>",
                                               "(Ljava/lang/String;)V"
                                                  );
                  if ( type_constructor_ID == NULL )
                     errexit( 
                     "Could not get Boolean( String s ) constructor."
                     "\n" );

                  /* get argument and convert into jstring */

                  if ( strcmp( parameter_type, "" ) == 0 )
                  {
                     sprintf( buf, "-80000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                  }

               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  argument = ( *env )->NewStringUTF( env, 
                                                     parameter_type );
                  if ( argument == NULL )
                     errexit( 
                           "Could not convert argument to jstring.\n" );

                  /* create Boolean */

                  next_argument = ( *env )->NewObject( env, type_class,
                                                    type_constructor_ID,
                                                    argument );
                  /* if argument conversion failed */
                  exception = ( * env )->ExceptionOccurred( env );
                  if ( exception )
                  {
                     ( *env )->ExceptionDescribe( env );
                     ( *env )->ExceptionClear( env );
                  }
                  if ( next_argument == NULL )
                  {
                     sprintf( buf, "-90000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                     continue;
                  }
              }

            }

            /* if parameter is a Character */

            else if ( strcmp( parameter_type, "-c" ) == 0 )
            {
               /* determine success or failure */
               parameter_type = strtok2( &pointer, &delimiter, " " );
               if ( strcmp( parameter_type, "Failed:" ) == 0 )
               {
                  sprintf( buf, "-260000" );
                  ( void ) write( slave_socket, buf, LINELEN );
                  instantiate_failed = JNI_TRUE;
               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  /* capture argument */
                  parameter_type = strtok2( &pointer, &delimiter, " " );

                  /* get class Char */
         
                  type_class = ( *env )->FindClass( env, 
                                         "java/lang/Character" );
                  if ( type_class == NULL )
                     errexit( "Could not find class Character.\n" );

                  /* get String constructor for new Char */

                  type_constructor_ID = ( *env )->GetMethodID( env,
                                                  type_class,
                                                  "<init>",
                                                  "(C)V"
                                                  );
                  if ( type_constructor_ID == NULL )
                     errexit( 
                     "Could not get Char( String s ) constructor.\n" );

                  if ( strcmp( parameter_type, "" ) == 0 )
                  {
                     sprintf( buf, "-100000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                  }

               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  char_argument = ( jchar ) *parameter_type;

                  /* create Character */

                  next_argument = ( *env )->NewObject( env, type_class,
                                                    type_constructor_ID,
                                                    char_argument );
                  /* if argument conversion failed */
                  exception = ( * env )->ExceptionOccurred( env );
                  if ( exception )
                  {
                     ( *env )->ExceptionDescribe( env );
                     ( *env )->ExceptionClear( env );
                  }
                  if ( next_argument == NULL )
                  {
                     sprintf( buf, "-110000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                     continue;
                  }
              }

            }

            /* if parameter is a Short */

            else if ( strcmp( parameter_type, "-s" ) == 0 )
            {
               /* determine success or failure */
               parameter_type = strtok2( &pointer, &delimiter, " " );
               if ( strcmp( parameter_type, "Failed:" ) == 0 )
               {
                  sprintf( buf, "-260000" );
                  ( void ) write( slave_socket, buf, LINELEN );
                  instantiate_failed = JNI_TRUE;
               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  /* capture argument */
                  parameter_type = strtok2( &pointer, &delimiter, " " );

                  /* get class Short */
         
                  type_class = ( *env )->FindClass( env, 
                                         "java/lang/Short" );
                  if ( type_class == NULL )
                     errexit( "Could not find class Short.\n" );

                  /* get String constructor for new Short */

                  type_constructor_ID = ( *env )->GetMethodID( env,
                                               type_class,
                                               "<init>",
                                               "(Ljava/lang/String;)V"
                                               );
                  if ( type_constructor_ID == NULL )
                     errexit( 
                     "Could not get Short( String s ) constructor.\n" );

                  /* get argument and convert into jstring */

                  if ( strcmp( parameter_type, "" ) == 0 )
                  {
                     sprintf( buf, "-120000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                  }

               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  argument = ( *env )->NewStringUTF( env, 
                                                     parameter_type );
                  if ( argument == NULL )
                     errexit( 
                           "Could not convert argument to jstring.\n" );

                  /* create Short */

                  next_argument = ( *env )->NewObject( env, type_class,
                                                    type_constructor_ID,
                                                    argument );
                  /* if argument conversion failed */
                  exception = ( * env )->ExceptionOccurred( env );
                  if ( exception )
                  {
                     ( *env )->ExceptionDescribe( env );
                     ( *env )->ExceptionClear( env );
                  }
                  if ( next_argument == NULL )
                  {
                     sprintf( buf, "-130000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                     continue;
                  }
              }

           }

           /* if parameter is an Int */

           else if ( strcmp( parameter_type, "-i" ) == 0 )
           {
               /* determine success or failure */
               parameter_type = strtok2( &pointer, &delimiter, " " );
               if ( strcmp( parameter_type, "Failed:" ) == 0 )
               {
                  sprintf( buf, "-260000" );
                  ( void ) write( slave_socket, buf, LINELEN );
                  instantiate_failed = JNI_TRUE;
               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  /* capture argument */
                  parameter_type = strtok2( &pointer, &delimiter, " " );

                  /* get class Integer */
         
                  type_class = ( *env )->FindClass( env, 
                                         "java/lang/Integer" );
                  if ( type_class == NULL )
                     errexit( "Could not find class Integer.\n" );

                  /* get String constructor for new Integer */

                  type_constructor_ID = ( *env )->GetMethodID( env,
                                                  type_class,
                                                  "<init>",
                                               "(Ljava/lang/String;)V"
                                                  );
                  if ( type_constructor_ID == NULL )
                     errexit( 
                        "Could not get Integer( String s ) constructor."
                        "\n" );

                  /* get argument and convert into jstring */

                  if ( strcmp( parameter_type, "" ) == 0 )
                  {
                     sprintf( buf, "-140000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                  }

               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  argument = ( *env )->NewStringUTF( env, 
                                                     parameter_type );
                  if ( argument == NULL )
                     errexit( 
                           "Could not convert argument to jstring.\n" );

                  /* create Integer */

                  next_argument = ( *env )->NewObject( env, type_class,
                                                    type_constructor_ID,
                                                    argument );
                  /* if argument conversion failed */
                  exception = ( * env )->ExceptionOccurred( env );
                  if ( exception )
                  {
                     ( *env )->ExceptionDescribe( env );
                     ( *env )->ExceptionClear( env );
                  }
                  if ( next_argument == NULL )
                  {
                     sprintf( buf, "-150000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                     continue;
                  }
              }

           }

           /* if parameter is a Long */

           else if ( strcmp( parameter_type, "-j" ) == 0 )
           {
               /* determine success or failure */
               parameter_type = strtok2( &pointer, &delimiter, " " );
               if ( strcmp( parameter_type, "Failed:" ) == 0 )
               {
                  sprintf( buf, "-260000" );
                  ( void ) write( slave_socket, buf, LINELEN );
                  instantiate_failed = JNI_TRUE;
               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  /* capture argument */
                  parameter_type = strtok2( &pointer, &delimiter, " " );

                  /* get class Long */
         
                  type_class = ( *env )->FindClass( env, 
                                         "java/lang/Long" );
                  if ( type_class == NULL )
                     errexit( "Could not find class Long.\n" );

                  /* get String constructor for new Long */

                  type_constructor_ID = ( *env )->GetMethodID( env,
                                               type_class,
                                               "<init>",
                                               "(Ljava/lang/String;)V"
                                               );
                  if ( type_constructor_ID == NULL )
                     errexit( 
                     "Could not get Long( String s ) constructor.\n" );

                  /* get argument and convert into jstring */

                  if ( strcmp( parameter_type, "" ) == 0 )
                  {
                     sprintf( buf, "-160000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                  }

               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  argument = ( *env )->NewStringUTF( env, 
                                                     parameter_type );
                  if ( argument == NULL )
                     errexit( 
                           "Could not convert argument to jstring.\n" );

                  /* create Long */

                  next_argument = ( *env )->NewObject( env, type_class,
                                                    type_constructor_ID,
                                                    argument );
                  /* if argument conversion failed */
                  exception = ( * env )->ExceptionOccurred( env );
                  if ( exception )
                  {
                     ( *env )->ExceptionDescribe( env );
                     ( *env )->ExceptionClear( env );
                  }
                  if ( next_argument == NULL )
                  {
                     sprintf( buf, "-170000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                     continue;
                  }
              }

           }

           /* if parameter is a Float */

           else if ( strcmp( parameter_type, "-f" ) == 0 )
           {
               /* determine success or failure */
               parameter_type = strtok2( &pointer, &delimiter, " " );
               if ( strcmp( parameter_type, "Failed:" ) == 0 )
               {
                  sprintf( buf, "-260000" );
                  ( void ) write( slave_socket, buf, LINELEN );
                  instantiate_failed = JNI_TRUE;
               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  /* capture argument */
                  parameter_type = strtok2( &pointer, &delimiter, " " );

                  /* get class Float */
         
                  type_class = ( *env )->FindClass( env, 
                                         "java/lang/Float" );
                  if ( type_class == NULL )
                     errexit( "Could not find class Float.\n" );

                  /* get String constructor for new Float */

                  type_constructor_ID = ( *env )->GetMethodID( env,
                                                  type_class,
                                                  "<init>",
                                                 "(Ljava/lang/String;)V"
                                                  );
                  if ( type_constructor_ID == NULL )
                     errexit( 
                     "Could not get Float( String s ) constructor.\n" );

                  /* get argument and convert into jstring */

                  if ( strcmp( parameter_type, "" ) == 0 )
                  {
                     sprintf( buf, "-180000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                  }

               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  argument = ( *env )->NewStringUTF( env, 
                                                     parameter_type );
                  if ( argument == NULL )
                     errexit( 
                           "Could not convert argument to jstring.\n" );

                  /* create Float */

                  next_argument = ( *env )->NewObject( env, type_class,
                                                    type_constructor_ID,
                                                    argument );
                  /* if argument conversion failed */
                  exception = ( * env )->ExceptionOccurred( env );
                  if ( exception )
                  {
                     ( *env )->ExceptionDescribe( env );
                     ( *env )->ExceptionClear( env );
                  }
                  if ( next_argument == NULL )
                  {
                     sprintf( buf, "-190000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                     continue;
                  }
              }

           }

           /* if parameter is a Double */

           else if ( strcmp( parameter_type, "-d" ) == 0 )
           {
               /* determine success or failure */
               parameter_type = strtok2( &pointer, &delimiter, " " );
               if ( strcmp( parameter_type, "Failed:" ) == 0 )
               {
                  sprintf( buf, "-260000" );
                  ( void ) write( slave_socket, buf, LINELEN );
                  instantiate_failed = JNI_TRUE;
               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  /* capture argument */
                  parameter_type = strtok2( &pointer, &delimiter, " " );

                  /* get class Double */
         
                  type_class = ( *env )->FindClass( env, 
                                         "java/lang/Double" );
                  if ( type_class == NULL )
                     errexit( "Could not find class Double.\n" );

                  /* get String constructor for new Double */

                  type_constructor_ID = ( *env )->GetMethodID( env,
                                                  type_class,
                                                  "<init>",
                                               "(Ljava/lang/String;)V"
                                                  );
                  if ( type_constructor_ID == NULL )
                     errexit( 
                     "Could not get Double( String s ) constructor."
                     "\n" );

                  /* get argument and convert into jstring */

                  if ( strcmp( parameter_type, "" ) == 0 )
                  {
                     sprintf( buf, "-200000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                  }

               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  argument = ( *env )->NewStringUTF( env, 
                                                     parameter_type );
                  if ( argument == NULL )
                     errexit( 
                           "Could not convert argument to jstring.\n" );

                  /* create Double */

                  next_argument = ( *env )->NewObject( env, type_class,
                                                    type_constructor_ID,
                                                    argument );
                  /* if argument conversion failed */
                  exception = ( * env )->ExceptionOccurred( env );
                  if ( exception )
                  {
                     ( *env )->ExceptionDescribe( env );
                     ( *env )->ExceptionClear( env );
                  }
                  if ( next_argument == NULL )
                  {
                     sprintf( buf, "-210000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                     continue;
                  }
              }

           }
           
           /* if parameter is a String */

           else if ( strcmp( parameter_type, "-t" ) == 0 )
           {
               /* determine success or failure */
               parameter_type = strtok2( &pointer, &delimiter, " " );
               if ( strcmp( parameter_type, "Failed:" ) == 0 )
               {
                  sprintf( buf, "-260000" );
                  ( void ) write( slave_socket, buf, LINELEN );
                  instantiate_failed = JNI_TRUE;
               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  /* capture argument */
                  parameter_type = strtok2( &pointer, &delimiter, 
                                            "\"" );

                  /* get argument and convert into jstring */

                  if ( strcmp( parameter_type, "" ) == 0 )
                  {
                     sprintf( buf, "-270000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                  }

               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  next_argument = ( *env )->NewStringUTF( env, 
                                                     parameter_type );
                  if ( next_argument == NULL )
                     errexit( 
                           "Could not convert argument to jstring.\n" );

              }

           }

           /* if parameter is an Object */

           else if ( strcmp( parameter_type, "-o" ) == 0 )
           {
               /* determine success or failure */
               parameter_type = strtok2( &pointer, &delimiter, " " );
               if ( strcmp( parameter_type, "Failed:" ) == 0 )
               {
                  sprintf( buf, "-280000" );
                  ( void ) write( slave_socket, buf, LINELEN );
                  instantiate_failed = JNI_TRUE;
               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  /* capture argument */
                  parameter_type = strtok2( &pointer, &delimiter, " " );

                  object_class = ( *env )->FindClass( env, 
                                         "java/lang/Object" );
                  if ( object_class == NULL )
                     errexit( "Could not find class Object.\n" );

                  /* get class Vector */
                  vector_class = ( *env )->FindClass( env,
                                          "java/util/Vector" );
                  if ( vector_class == NULL )
                     errexit( "Could not find class Vector.\n" );

                  /* get getObjectVector() method */
                  get_object_vector_ID = ( *env )->GetMethodID( env,
                                                   pjvm_cls,
                                                   "getObjectVector",
                                                "()Ljava/util/Vector;"
                                                   );

                  if ( get_object_vector_ID == NULL )
                     errexit( "Could not find pjvm.getObjectVector()."
                              "\n" );

                  object_vector = ( jvector ) ( *env )->
                                              CallObjectMethod( 
                                              env, pjvm,
                                              get_object_vector_ID );

                  /* get Vector.size() */
                  vector_size_ID = ( *env )->GetMethodID( env,
                                                          vector_class,
                                                          "size",
                                                          "()I" );
                  if ( vector_size_ID == NULL )
                     errexit( "Could not get Vector.size()\n" );

               
                  /* get size of Object vector */
                  vector_size_2 = ( int ) ( *env )->CallIntMethod( env,
                                                  object_vector,
                                                  vector_size_ID );
               
                  if ( atoi( parameter_type ) > ( vector_size_2 - 1 ) )
                  {
                     sprintf( buf, "-230000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                  }

               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  /* get Vector.elementAt() */
                  vector_element_at_ID = ( *env )->GetMethodID( env,
                                                vector_class,
                                                "elementAt",
                                                "(I)Ljava/lang/Object;"
                                                );
                  if ( vector_element_at_ID == NULL )
                     errexit( "Could not find Vector.elementAt().\n" );
                  
                  /* create Object */

                  next_argument = ( *env )->CallObjectMethod( env, 
                                               object_vector,
                                               vector_element_at_ID,
                                               atoi( parameter_type ) );

                  /* if argument conversion failed */
                  exception = ( * env )->ExceptionOccurred( env );
                  if ( exception )
                  {
                     ( *env )->ExceptionDescribe( env );
                     ( *env )->ExceptionClear( env );
                  }
                  if ( next_argument == NULL )
                  {
                     sprintf( buf, "-240000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                     continue;
                  }
              }

           }
 
           if ( instantiate_failed == JNI_FALSE )
           {
               vector_add_ID = ( *env )->GetMethodID( env, 
                                         vector_class,
                                         "add",
                                         "(Ljava/lang/Object;)Z" );
               if ( vector_add_ID == NULL )
                  errexit( "Could not get Vector.add().\n" );

               add_successful = ( *env )->CallBooleanMethod( env,
                                   ( jobject ) argument_vector, 
                                   vector_add_ID,
                                   next_argument );

               if ( add_successful == JNI_FALSE )
                  errexit( "Could not add argument to Vector." );

            }
         } /* end of for loop */

         /* attempt to instantiate the object */

         /* get pjvm.loadObject() */

         if ( instantiate_failed == JNI_FALSE )
         {
            load_object_ID = ( *env )->GetMethodID( env, pjvm_cls,
                                    "loadObject",
                                    "(Ljava/lang/reflect/Constructor;"
                                    "Ljava/util/Vector;Z)Z" );
            if ( load_object_ID == NULL )
               errexit( "Could not get pjvm.loadObject()\n" );

            instantiate_failed = ( *env )->CallBooleanMethod( env,
                                              pjvm, load_object_ID,
                                              class_constructor,
                                              argument_vector,
                                              garbage_collected );
         }

         /* if object creation failed */
         if ( instantiate_failed == JNI_FALSE )
         {
            sprintf( buf, "-250000" );
            ( void ) write( slave_socket, buf, LINELEN );
         }
         else
         {
            /* get class Vector */
            vector_class = ( *env )->FindClass( env,
                                       "java/util/Vector" );
            if ( vector_class == NULL )
               errexit( "Could not find class Vector.\n" );

            /* get getObjectVector() method */
            get_object_vector_ID = ( *env )->GetMethodID( env,
                                             pjvm_cls,
                                             "getObjectVector",
                                             "()Ljava/util/Vector;"
                                             );

            if ( get_object_vector_ID == NULL )
               errexit( "Could not find pjvm.getObjectVector().\n" );

               
            object_vector = ( jvector ) ( *env )->CallObjectMethod( 
                                              env, pjvm,
                                              get_object_vector_ID );

            /* get Vector.size() */
            vector_size_ID = ( *env )->GetMethodID( env,
                                                    vector_class,
                                                    "size",
                                                    "()I" );
            if ( vector_size_ID == NULL )
               errexit( "Could not get Vector.size()\n" );

               
            /* get size of Object vector */
            vector_size = ( int ) ( *env )->CallIntMethod( env,
                                            object_vector,
                                            vector_size_ID );
 
            sprintf( buf, "%d", vector_size - 1 );
            ( void ) write( slave_socket, buf, LINELEN );

            /* if this is final link in chain, call finalizer */
            if ( call_finalizer == TRUE )
            {
               /* get pjvm.collectGarbage() */
               collect_garbage_ID = ( *env )->GetMethodID( env,
                                              pjvm_cls,
                                              "collectGarbage",
                                              "()V" );
               if ( collect_garbage_ID == NULL )
                  errexit( "Could not get pjvm.collectGarbage().\n" );

               /* call collectGarbage() */
               ( *env )->CallVoidMethod( env, pjvm, collect_garbage_ID
                                       );
            }
         }

         /* reset argument count */
         arg_count = 0;
      }

      /* if command comes from callmethod */

      else if ( strncmp( buf, "clm", 3 ) == 0 )
      {
         /* clear largebuf */

         strcpy( largebuf, "" );

         /* discard the first two tokens */
         strtok2( &pointer, &delimiter, " " );
         strtok2( &pointer, &delimiter, " " );

         /* get selected class number */

         selected_class_number = atoi( strtok2( &pointer, &delimiter,
                                                " " ) );

         /* get selected method number */
         selected_method_number = atoi( strtok2( &pointer,
                                       &delimiter, " " ) );

         /* determine if method is static */
         if ( atoi( strtok2( &pointer, &delimiter, " " ) ) == TRUE )
            static_method = JNI_TRUE;
         else
            static_method = JNI_FALSE;

         /* if the method is not static, get the selected object num */
         if ( static_method == JNI_FALSE )
            selected_object_number = atoi( strtok2( &pointer,
                                           &delimiter, " " ) );

         /* determine if method number is from declared or total
            methods */
         if ( atoi( strtok2( &pointer, &delimiter, " " ) ) == TRUE )
            declared_method = JNI_TRUE;
         else
            declared_method = JNI_FALSE;

         /* determine if object will be garbage collected */
         if ( atoi( strtok2( &pointer, &delimiter, " " ) ) == TRUE )
            garbage_collected = JNI_TRUE;
         else
            garbage_collected = JNI_FALSE;

         /* determine if this is the final object in a chain */
         call_finalizer = atoi( strtok2( &pointer, &delimiter, " " ) );

         /* Get getClassVector() method */

         get_class_vector_ID = ( *env )->GetMethodID( env, pjvm_cls,
                               "getClassVector", 
                               "()Ljava/util/Vector;" );
         if ( get_class_vector_ID == NULL )
            errexit( "Can't find pjvm.getClassVector().\n" );

         /* get vector of classes */

         class_vector = ( *env )->CallObjectMethod( env, pjvm,
                                                 get_class_vector_ID );
         if ( class_vector == NULL )
            errexit( "Could not retrieve record of loaded classes.\n" );

         /* get Vector class */

         vector_class = ( *env )->FindClass( env, "java/util/Vector" );
         if ( vector_class == NULL )
            errexit( "Could not find class Vector.\n" );

         /* get Vector.size() */

         vector_size_ID = ( *env )->GetMethodID( env, vector_class, 
                                                 "size", "()I" );
         if ( vector_size_ID == NULL )
            errexit( "Could not get Vector.size().\n" );
      
         /* get size of class vector */

         vector_size = ( int ) ( *env )->CallIntMethod( env,
                                                       class_vector,
                                                       vector_size_ID );

         /* get Vector.elementAt() */

         vector_element_at_ID = ( *env )->GetMethodID( env, 
                                          vector_class,
                                          "elementAt",
                                          "(I)Ljava/lang/Object;" );
         if ( vector_element_at_ID == NULL )
            errexit( "Could not get Vector.elementAt().\n" );

         /* get Object class */

         object_class = ( *env )->FindClass( env, "java/lang/Object" );
         if ( object_class == NULL )
            errexit( "Could not find class Object.\n" );

         /* get size of class vector */

         vector_size = ( int ) ( *env )->CallIntMethod( env,
                                         class_vector, vector_size_ID );

         if ( selected_class_number > ( vector_size - 1 ) )
         {
            sprintf( largebuf, "-20000" );
            ( void ) write( slave_socket, largebuf, BUFSIZE );
            continue;
         }

         /* get the selected class */

         selected_class = ( *env )->CallObjectMethod( env, class_vector,
                                    vector_element_at_ID, ( jint )
                                    selected_class_number );
         if ( selected_class == NULL )
            errexit( "Could not retrieve selected class.\n" );

         /* if method is from declared methods, get
            pjvm.getDeclaredMethods() */
         if ( declared_method == JNI_TRUE )
         {
            get_methods_ID = ( *env )->GetMethodID( env,
                                                pjvm_cls,
                                                "getDeclaredMethods",
                                                "(Ljava/lang/Class;)"
                                                "Ljava/util/Vector;" );
            if ( get_methods_ID == NULL )
               errexit( "Could not get pjvm.getDeclaredMethods().\n" );
         }
         
         /* otherwise, get pjvm.getMethods() */
         else
         {
            get_methods_ID = ( *env )->GetMethodID( env,
                                               pjvm_cls,
                                               "getDeclaredMethods",
                                               "(Ljava/lang/Class;)"
                                               "Ljava/util/Vector;" );
            if ( get_methods_ID == NULL )
               errexit( "Could not get pjvm.getMethods().\n" );

         }

         /* get methods */

         method_vector = ( *env )->CallObjectMethod( env,
                                              pjvm, get_methods_ID,
                                              selected_class );
         if ( class_constructor_vector == NULL )
            errexit( "Could not get class constructors.\n" );

         /* get size of method vector */

         vector_size = ( int ) ( *env )->CallIntMethod( env,
                                         method_vector,
                                         vector_size_ID );

         if ( selected_method_number > ( vector_size - 1 ) )
         {
            sprintf( largebuf, "-30000" );
            ( void ) write( slave_socket, largebuf, BUFSIZE );
            continue;
         }

         /* attempt to get selected method */

         selected_method = ( *env )->CallObjectMethod( env,
                                     method_vector,
                                     vector_element_at_ID, ( jint )
                                     selected_method_number );
         if ( selected_method == NULL )
            errexit( "Could not read method vector.\n" );

         /* if method is not static, get objects as well */
         if ( static_method == JNI_FALSE )
         {
            /* get pjvm.getObjectVector() */
            get_object_vector_ID = ( *env )->GetMethodID( env,
                                             pjvm_cls,
                                             "getObjectVector",
                                             "()Ljava/util/Vector;" );
            if ( get_object_vector_ID == NULL )
               errexit( "Could not get pjvm.getObjectVector().\n" );
            
            /* get objects */
            object_vector = ( *env )->CallObjectMethod( env,
                                           pjvm, get_object_vector_ID );
            if ( object_vector == NULL )
               errexit( "Could not get objects.\n" );

            /* get size of object vector */
            vector_size = ( int ) ( *env )->CallIntMethod( env,
                                            object_vector,
                                            vector_size_ID );

            if ( selected_object_number > ( vector_size - 1 ) )
            {
               sprintf( largebuf, "-35000" );
               ( void ) write( slave_socket, largebuf, BUFSIZE );
               continue;
            }

            /* get selected object */
            selected_object_array = ( *env )->CallObjectMethod( env,
                                        object_vector,
                                        vector_element_at_ID, ( jint )
                                        selected_object_number );
            if ( selected_object_array == NULL )
               errexit( "Error reading object vector.\n" );

            selected_object = ( *env )->GetObjectArrayElement( env,
                                        selected_object_array,
                                        ( jsize ) 0 );
            if ( selected_object == NULL )
               errexit( "Error reading Object array.\n" );

         }

         /* get pjvm.getParameters() for Methods */

         get_parameters_ID = ( *env )->GetMethodID( env, pjvm_cls,
                                     "getParameters",
                                     "(Ljava/lang/reflect/Method;)"
                                     "Ljava/util/Vector;" );
         if ( get_parameters_ID == NULL )
            errexit( "Could not get pjvm.getParameters().\n" );


         /* get method parameters */

         method_parameter_vector = ( *env )->CallObjectMethod( env,
                                        pjvm, get_parameters_ID,
                                        selected_method );
         if ( constructor_parameter_vector == NULL )
            errexit( "Could not get method parameters.\n" );

         /* get size of parameter vector */

         vector_size = ( int ) ( *env )->CallIntMethod( env,
                                         method_parameter_vector,
                                         vector_size_ID );

         sprintf( buf, "0" );
         ( void ) write( slave_socket, buf, LINELEN );


         /* if everything has worked out, get parameters from user */

         for ( cc = 0; cc < BUFSIZE; cc += n )
         {
            n = read( slave_socket, largebuf, BUFSIZE );
            if ( n < 0 )
               errexit( "read: %s\n", strerror( errno ) );
         }

         if ( ( vector_size > 0 ) && ( strcmp( largebuf, "" ) == 0 ) )
         {
            sprintf( largebuf, "-40000" );
            ( void ) write( slave_socket, largebuf, BUFSIZE );
            continue;
         }

         /* get empty Vector constructor */

         vector_constructor_ID = ( *env )->GetMethodID( env,
                                           vector_class, "<init>",
                                           "()V" );
         if ( vector_constructor_ID == NULL )
            errexit( "Could not get constructor for class Vector.\n" );

         /* create argument vector */

         argument_vector = ( *env )->NewObject( env, vector_class,
                                                vector_constructor_ID );
         if ( argument_vector == NULL )
            errexit( "Could not create argument Vector.\n" );

         /* populate argument vector */

         pointer = largebuf;


         for ( i = 0; i < vector_size; i++ )
         {
            jobject next_argument;
            jclass type_class;
            jmethodID type_constructor_ID;
            jstring argument;
            jmethodID vector_add_ID;
            jboolean add_successful;
            char *parameter_type;
            jchar char_argument;

            parameter_type = strtok2( &pointer, &delimiter, " " );

            if ( strcmp( parameter_type, "" ) == 0 )
            {
               if ( i < ( vector_size - 1 ) )
                  sprintf( buf, "-50000" );
               else
                  sprintf( buf, "-220000" );
               ( void ) write( slave_socket, buf, LINELEN );
               instantiate_failed = JNI_TRUE;
               continue;
            }

            /* if parameter is a Byte */

            if ( strcmp( parameter_type, "-b" ) == 0 )
            {
               /* determine success or failure */
               parameter_type = strtok2( &pointer, &delimiter, " " );
               if ( strcmp( parameter_type, "Failed:" ) == 0 )
               {
                  sprintf( buf, "-260000" );
                  ( void ) write( slave_socket, buf, LINELEN );
                  instantiate_failed = JNI_TRUE;
               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  /* capture argument */
                  parameter_type = strtok2( &pointer, &delimiter, " " );

                  /* get class Byte */
         
                  type_class = ( *env )->FindClass( env, 
                                      "java/lang/Byte" );
                  if ( type_class == NULL )
                     errexit( "Could not find class Byte.\n" );

                  /* get String constructor for new Byte */

                  type_constructor_ID = ( *env )->GetMethodID( env,
                                               type_class,
                                               "<init>",
                                               "(Ljava/lang/String;)V"
                                               );
                  if ( type_constructor_ID == NULL )
                     errexit( "Could not get Byte( String s ) "
                              "constructor. \n" );

                  /* get argument and convert into jstring */

                  if ( strcmp( parameter_type, "" ) == 0 )
                  {
                     sprintf( buf, "-60000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                  }

               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  argument = ( *env )->NewStringUTF( env, 
                                                     parameter_type );
                  if ( argument == NULL )
                     errexit( 
                           "Could not convert argument to jstring.\n" );

                  /* create Byte */

                  next_argument = ( *env )->NewObject( env, type_class,
                                                    type_constructor_ID,
                                                    argument );
                  /* if argument conversion failed */
                  exception = ( * env )->ExceptionOccurred( env );
                  if ( exception )
                  {
                     ( *env )->ExceptionDescribe( env );
                     ( *env )->ExceptionClear( env );
                  }
                  if ( next_argument == NULL )
                  {
                     sprintf( buf, "-70000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                     continue;
                  }
              }
            }

            /* if parameter is a Boolean */

            else if ( strcmp( parameter_type, "-z" ) == 0 )
            {
               /* determine success or failure */
               parameter_type = strtok2( &pointer, &delimiter, " " );
               if ( strcmp( parameter_type, "Failed:" ) == 0 )
               {
                  sprintf( buf, "-260000" );
                  ( void ) write( slave_socket, buf, LINELEN );
                  instantiate_failed = JNI_TRUE;
               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  /* capture argument */
                  parameter_type = strtok2( &pointer, &delimiter, " " );

                  /* get class Boolean */
         
                  type_class = ( *env )->FindClass( env, 
                                         "java/lang/Boolean" );
                  if ( type_class == NULL )
                     errexit( "Could not find class Boolean.\n" );

                  /* get String constructor for new Boolean */

                  type_constructor_ID = ( *env )->GetMethodID( env,
                                                  type_class,
                                                  "<init>",
                                               "(Ljava/lang/String;)V"
                                                  );
                  if ( type_constructor_ID == NULL )
                     errexit( 
                     "Could not get Boolean( String s ) constructor."
                     "\n" );

                  /* get argument and convert into jstring */

                  if ( strcmp( parameter_type, "" ) == 0 )
                  {
                     sprintf( buf, "-80000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                  }

               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  argument = ( *env )->NewStringUTF( env, 
                                                     parameter_type );
                  if ( argument == NULL )
                     errexit( 
                           "Could not convert argument to jstring.\n" );

                  /* create Boolean */

                  next_argument = ( *env )->NewObject( env, type_class,
                                                    type_constructor_ID,
                                                    argument );
                  /* if argument conversion failed */
                  exception = ( * env )->ExceptionOccurred( env );
                  if ( exception )
                  {
                     ( *env )->ExceptionDescribe( env );
                     ( *env )->ExceptionClear( env );
                  }
                  if ( next_argument == NULL )
                  {
                     sprintf( buf, "-90000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                     continue;
                  }
              }

            }

            /* if parameter is a Character */

            else if ( strcmp( parameter_type, "-c" ) == 0 )
            {
               /* determine success or failure */
               parameter_type = strtok2( &pointer, &delimiter, " " );
               if ( strcmp( parameter_type, "Failed:" ) == 0 )
               {
                  sprintf( buf, "-260000" );
                  ( void ) write( slave_socket, buf, LINELEN );
                  instantiate_failed = JNI_TRUE;
               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  /* capture argument */
                  parameter_type = strtok2( &pointer, &delimiter, " " );

                  /* get class Char */
         
                  type_class = ( *env )->FindClass( env, 
                                         "java/lang/Character" );
                  if ( type_class == NULL )
                     errexit( "Could not find class Character.\n" );

                  /* get String constructor for new Char */

                  type_constructor_ID = ( *env )->GetMethodID( env,
                                                  type_class,
                                                  "<init>",
                                                  "(C)V"
                                                  );
                  if ( type_constructor_ID == NULL )
                     errexit( 
                     "Could not get Char( String s ) constructor.\n" );

                  if ( strcmp( parameter_type, "" ) == 0 )
                  {
                     sprintf( buf, "-100000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                  }

               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  char_argument = ( jchar ) *parameter_type;

                  /* create Character */

                  next_argument = ( *env )->NewObject( env, type_class,
                                                    type_constructor_ID,
                                                    char_argument );
                  /* if argument conversion failed */
                  exception = ( * env )->ExceptionOccurred( env );
                  if ( exception )
                  {
                     ( *env )->ExceptionDescribe( env );
                     ( *env )->ExceptionClear( env );
                  }
                  if ( next_argument == NULL )
                  {
                     sprintf( buf, "-110000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                     continue;
                  }
              }

            }

            /* if parameter is a Short */

            else if ( strcmp( parameter_type, "-s" ) == 0 )
            {
               /* determine success or failure */
               parameter_type = strtok2( &pointer, &delimiter, " " );
               if ( strcmp( parameter_type, "Failed:" ) == 0 )
               {
                  sprintf( buf, "-260000" );
                  ( void ) write( slave_socket, buf, LINELEN );
                  instantiate_failed = JNI_TRUE;
               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  /* capture argument */
                  parameter_type = strtok2( &pointer, &delimiter, " " );

                  /* get class Short */
         
                  type_class = ( *env )->FindClass( env, 
                                         "java/lang/Short" );
                  if ( type_class == NULL )
                     errexit( "Could not find class Short.\n" );

                  /* get String constructor for new Short */

                  type_constructor_ID = ( *env )->GetMethodID( env,
                                               type_class,
                                               "<init>",
                                               "(Ljava/lang/String;)V"
                                               );
                  if ( type_constructor_ID == NULL )
                     errexit( 
                     "Could not get Short( String s ) constructor.\n" );

                  /* get argument and convert into jstring */

                  if ( strcmp( parameter_type, "" ) == 0 )
                  {
                     sprintf( buf, "-120000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                  }

               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  argument = ( *env )->NewStringUTF( env, 
                                                     parameter_type );
                  if ( argument == NULL )
                     errexit( 
                           "Could not convert argument to jstring.\n" );

                  /* create Short */

                  next_argument = ( *env )->NewObject( env, type_class,
                                                    type_constructor_ID,
                                                    argument );
                  /* if argument conversion failed */
                  exception = ( * env )->ExceptionOccurred( env );
                  if ( exception )
                  {
                     ( *env )->ExceptionDescribe( env );
                     ( *env )->ExceptionClear( env );
                  }
                  if ( next_argument == NULL )
                  {
                     sprintf( buf, "-130000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                     continue;
                  }
              }

           }

           /* if parameter is an Int */

           else if ( strcmp( parameter_type, "-i" ) == 0 )
           {
               /* determine success or failure */
               parameter_type = strtok2( &pointer, &delimiter, " " );
               if ( strcmp( parameter_type, "Failed:" ) == 0 )
               {
                  sprintf( buf, "-260000" );
                  ( void ) write( slave_socket, buf, LINELEN );
                  instantiate_failed = JNI_TRUE;
               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  /* capture argument */
                  parameter_type = strtok2( &pointer, &delimiter, " " );

                  /* get class Integer */
         
                  type_class = ( *env )->FindClass( env, 
                                         "java/lang/Integer" );
                  if ( type_class == NULL )
                     errexit( "Could not find class Integer.\n" );

                  /* get String constructor for new Integer */

                  type_constructor_ID = ( *env )->GetMethodID( env,
                                                  type_class,
                                                  "<init>",
                                               "(Ljava/lang/String;)V"
                                                  );
                  if ( type_constructor_ID == NULL )
                     errexit( 
                        "Could not get Integer( String s ) constructor."
                        "\n" );

                  /* get argument and convert into jstring */

                  if ( strcmp( parameter_type, "" ) == 0 )
                  {
                     sprintf( buf, "-140000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                  }

               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  argument = ( *env )->NewStringUTF( env, 
                                                     parameter_type );
                  if ( argument == NULL )
                     errexit( 
                           "Could not convert argument to jstring.\n" );

                  /* create Integer */

                  next_argument = ( *env )->NewObject( env, type_class,
                                                    type_constructor_ID,
                                                    argument );
                  /* if argument conversion failed */
                  exception = ( * env )->ExceptionOccurred( env );
                  if ( exception )
                  {
                     ( *env )->ExceptionDescribe( env );
                     ( *env )->ExceptionClear( env );
                  }
                  if ( next_argument == NULL )
                  {
                     sprintf( buf, "-150000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                     continue;
                  }
              }

           }

           /* if parameter is a Long */

           else if ( strcmp( parameter_type, "-j" ) == 0 )
           {
               /* determine success or failure */
               parameter_type = strtok2( &pointer, &delimiter, " " );
               if ( strcmp( parameter_type, "Failed:" ) == 0 )
               {
                  sprintf( buf, "-260000" );
                  ( void ) write( slave_socket, buf, LINELEN );
                  instantiate_failed = JNI_TRUE;
               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  /* capture argument */
                  parameter_type = strtok2( &pointer, &delimiter, " " );

                  /* get class Long */
         
                  type_class = ( *env )->FindClass( env, 
                                         "java/lang/Long" );
                  if ( type_class == NULL )
                     errexit( "Could not find class Long.\n" );

                  /* get String constructor for new Long */

                  type_constructor_ID = ( *env )->GetMethodID( env,
                                               type_class,
                                               "<init>",
                                               "(Ljava/lang/String;)V"
                                               );
                  if ( type_constructor_ID == NULL )
                     errexit( 
                     "Could not get Long( String s ) constructor.\n" );

                  /* get argument and convert into jstring */

                  if ( strcmp( parameter_type, "" ) == 0 )
                  {
                     sprintf( buf, "-160000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                  }

               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  argument = ( *env )->NewStringUTF( env, 
                                                     parameter_type );
                  if ( argument == NULL )
                     errexit( 
                           "Could not convert argument to jstring.\n" );

                  /* create Long */

                  next_argument = ( *env )->NewObject( env, type_class,
                                                    type_constructor_ID,
                                                    argument );
                  /* if argument conversion failed */
                  exception = ( * env )->ExceptionOccurred( env );
                  if ( exception )
                  {
                     ( *env )->ExceptionDescribe( env );
                     ( *env )->ExceptionClear( env );
                  }
                  if ( next_argument == NULL )
                  {
                     sprintf( buf, "-170000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                     continue;
                  }
              }

           }

           /* if parameter is a Float */

           else if ( strcmp( parameter_type, "-f" ) == 0 )
           {
               /* determine success or failure */
               parameter_type = strtok2( &pointer, &delimiter, " " );
               if ( strcmp( parameter_type, "Failed:" ) == 0 )
               {
                  sprintf( buf, "-260000" );
                  ( void ) write( slave_socket, buf, LINELEN );
                  instantiate_failed = JNI_TRUE;
               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  /* capture argument */
                  parameter_type = strtok2( &pointer, &delimiter, " " );

                  /* get class Float */
         
                  type_class = ( *env )->FindClass( env, 
                                         "java/lang/Float" );
                  if ( type_class == NULL )
                     errexit( "Could not find class Float.\n" );

                  /* get String constructor for new Float */

                  type_constructor_ID = ( *env )->GetMethodID( env,
                                                  type_class,
                                                  "<init>",
                                                 "(Ljava/lang/String;)V"
                                                  );
                  if ( type_constructor_ID == NULL )
                     errexit( 
                     "Could not get Float( String s ) constructor.\n" );

                  /* get argument and convert into jstring */

                  if ( strcmp( parameter_type, "" ) == 0 )
                  {
                     sprintf( buf, "-180000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                  }

               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  argument = ( *env )->NewStringUTF( env, 
                                                     parameter_type );
                  if ( argument == NULL )
                     errexit( 
                           "Could not convert argument to jstring.\n" );

                  /* create Float */

                  next_argument = ( *env )->NewObject( env, type_class,
                                                    type_constructor_ID,
                                                    argument );
                  /* if argument conversion failed */
                  exception = ( * env )->ExceptionOccurred( env );
                  if ( exception )
                  {
                     ( *env )->ExceptionDescribe( env );
                     ( *env )->ExceptionClear( env );
                  }
                  if ( next_argument == NULL )
                  {
                     sprintf( buf, "-190000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                     continue;
                  }
              }

           }

           /* if parameter is a Double */

           else if ( strcmp( parameter_type, "-d" ) == 0 )
           {
               /* determine success or failure */
               parameter_type = strtok2( &pointer, &delimiter, " " );
               if ( strcmp( parameter_type, "Failed:" ) == 0 )
               {
                  sprintf( buf, "-260000" );
                  ( void ) write( slave_socket, buf, LINELEN );
                  instantiate_failed = JNI_TRUE;
               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  /* capture argument */
                  parameter_type = strtok2( &pointer, &delimiter, " " );

                  /* get class Double */
         
                  type_class = ( *env )->FindClass( env, 
                                         "java/lang/Double" );
                  if ( type_class == NULL )
                     errexit( "Could not find class Double.\n" );

                  /* get String constructor for new Double */

                  type_constructor_ID = ( *env )->GetMethodID( env,
                                                  type_class,
                                                  "<init>",
                                               "(Ljava/lang/String;)V"
                                                  );
                  if ( type_constructor_ID == NULL )
                     errexit( 
                     "Could not get Double( String s ) constructor."
                     "\n" );

                  /* get argument and convert into jstring */

                  if ( strcmp( parameter_type, "" ) == 0 )
                  {
                     sprintf( buf, "-200000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                  }

               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  argument = ( *env )->NewStringUTF( env, 
                                                     parameter_type );
                  if ( argument == NULL )
                     errexit( 
                           "Could not convert argument to jstring.\n" );

                  /* create Double */

                  next_argument = ( *env )->NewObject( env, type_class,
                                                    type_constructor_ID,
                                                    argument );
                  /* if argument conversion failed */
                  exception = ( * env )->ExceptionOccurred( env );
                  if ( exception )
                  {
                     ( *env )->ExceptionDescribe( env );
                     ( *env )->ExceptionClear( env );
                  }
                  if ( next_argument == NULL )
                  {
                     sprintf( buf, "-210000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                     continue;
                  }
              }

           }
           
           /* if parameter is a String */

           else if ( strcmp( parameter_type, "-t" ) == 0 )
           {
               /* determine success or failure */
               parameter_type = strtok2( &pointer, &delimiter, " " );
               if ( strcmp( parameter_type, "Failed:" ) == 0 )
               {
                  sprintf( buf, "-260000" );
                  ( void ) write( slave_socket, buf, LINELEN );
                  instantiate_failed = JNI_TRUE;
               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  /* capture argument */
                  parameter_type = strtok2( &pointer, &delimiter, 
                                            "\"" );

                  /* get argument and convert into jstring */

                  if ( strcmp( parameter_type, "" ) == 0 )
                  {
                     sprintf( buf, "-270000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                  }

               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  next_argument = ( *env )->NewStringUTF( env, 
                                                     parameter_type );
                  if ( next_argument == NULL )
                     errexit( 
                           "Could not convert argument to jstring.\n" );

              }

           }

           /* if parameter is an Object */

           else if ( strcmp( parameter_type, "-o" ) == 0 )
           {
               /* determine success or failure */
               parameter_type = strtok2( &pointer, &delimiter, " " );
               if ( strcmp( parameter_type, "Failed:" ) == 0 )
               {
                  sprintf( buf, "-280000" );
                  ( void ) write( slave_socket, buf, LINELEN );
                  instantiate_failed = JNI_TRUE;
               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  /* capture argument */
                  parameter_type = strtok2( &pointer, &delimiter, " " );

                  object_class = ( *env )->FindClass( env, 
                                         "java/lang/Object" );
                  if ( object_class == NULL )
                     errexit( "Could not find class Object.\n" );

                  /* get class Vector */
                  vector_class = ( *env )->FindClass( env,
                                          "java/util/Vector" );
                  if ( vector_class == NULL )
                     errexit( "Could not find class Vector.\n" );

                  /* get getObjectVector() method */
                  get_object_vector_ID = ( *env )->GetMethodID( env,
                                                   pjvm_cls,
                                                   "getObjectVector",
                                                "()Ljava/util/Vector;"
                                                   );

                  if ( get_object_vector_ID == NULL )
                     errexit( "Could not find pjvm.getObjectVector()."
                              "\n" );

                  object_vector = ( jvector ) ( *env )->
                                              CallObjectMethod( 
                                              env, pjvm,
                                              get_object_vector_ID );

                  /* get Vector.size() */
                  vector_size_ID = ( *env )->GetMethodID( env,
                                                          vector_class,
                                                          "size",
                                                          "()I" );
                  if ( vector_size_ID == NULL )
                     errexit( "Could not get Vector.size()\n" );

               
                  /* get size of Object vector */
                  vector_size_2 = ( int ) ( *env )->CallIntMethod( env,
                                                  object_vector,
                                                  vector_size_ID );
               
                  if ( atoi( parameter_type ) > ( vector_size_2 - 1 ) )
                  {
                     sprintf( buf, "-230000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                  }

               }

               if ( instantiate_failed == JNI_FALSE )
               {
                  /* get Vector.elementAt() */
                  vector_element_at_ID = ( *env )->GetMethodID( env,
                                                vector_class,
                                                "elementAt",
                                                "(I)Ljava/lang/Object;"
                                                );
                  if ( vector_element_at_ID == NULL )
                     errexit( "Could not find Vector.elementAt().\n" );
                  
                  /* create Object */

                  next_argument = ( *env )->CallObjectMethod( env, 
                                               object_vector,
                                               vector_element_at_ID,
                                               atoi( parameter_type ) );

                  /* if argument conversion failed */
                  exception = ( * env )->ExceptionOccurred( env );
                  if ( exception )
                  {
                     ( *env )->ExceptionDescribe( env );
                     ( *env )->ExceptionClear( env );
                  }
                  if ( next_argument == NULL )
                  {
                     sprintf( buf, "-240000" );
                     ( void ) write( slave_socket, buf, LINELEN );
                     instantiate_failed = JNI_TRUE;
                     continue;
                  }
              }

           }
 
           if ( instantiate_failed == JNI_FALSE )
           {
               vector_add_ID = ( *env )->GetMethodID( env, 
                                         vector_class,
                                         "add",
                                         "(Ljava/lang/Object;)Z" );
               if ( vector_add_ID == NULL )
                  errexit( "Could not get Vector.add().\n" );

               add_successful = ( *env )->CallBooleanMethod( env,
                                   ( jobject ) argument_vector, 
                                   vector_add_ID,
                                   next_argument );

               if ( add_successful == JNI_FALSE )
                  errexit( "Could not add argument to Vector." );

            }
         } /* end of for loop */

         /* attempt to instantiate the object */


         if ( instantiate_failed == JNI_FALSE )
         {
            /* get pjvm.executeMethod() */
            execute_method_ID = ( *env )->GetMethodID( env, pjvm_cls,
                                          "executeMethod",
                                          "(Ljava/lang/reflect/Method;"
                                          "Ljava/util/Vector;"
                                          "Ljava/lang/Object;)"
                                          "Ljava/lang/Object;" );
            if ( execute_method_ID == NULL )
               errexit( "Could not get pjvm.executeMethod().\n" );

            /* get pjvm.storeObject() */
            store_object_ID = ( *env )->GetMethodID( env, pjvm_cls,
                                        "storeObject",
                                        "(Ljava/lang/Object;Z)Z" );
            if ( store_object_ID == NULL )
               errexit( "Could not get pjvm.storeObject().\n" );

            if ( static_method == JNI_FALSE )
               loaded_object = ( *env )->CallObjectMethod( env,
                                             pjvm, execute_method_ID,
                                             selected_method,
                                             argument_vector,
                                             selected_object );
            else
               loaded_object = ( *env )->CallObjectMethod( env,
                                             pjvm, execute_method_ID,
                                             selected_method,
                                             argument_vector,
                                             NULL );

            instantiate_failed = ( *env )->CallBooleanMethod( env,
                                              pjvm, store_object_ID,
                                              loaded_object,
                                              garbage_collected );
         }

         /* if object creation failed */
         if ( instantiate_failed == JNI_FALSE )
         {
            sprintf( buf, "-250000" );
            ( void ) write( slave_socket, buf, LINELEN );
         }
         else
         {
            /* get class Vector */
            vector_class = ( *env )->FindClass( env,
                                       "java/util/Vector" );
            if ( vector_class == NULL )
               errexit( "Could not find class Vector.\n" );

            /* get getObjectVector() method */
            get_object_vector_ID = ( *env )->GetMethodID( env,
                                             pjvm_cls,
                                             "getObjectVector",
                                             "()Ljava/util/Vector;"
                                             );

            if ( get_object_vector_ID == NULL )
               errexit( "Could not find pjvm.getObjectVector().\n" );

               
            object_vector = ( jvector ) ( *env )->CallObjectMethod( 
                                              env, pjvm,
                                              get_object_vector_ID );

            /* get Vector.size() */
            vector_size_ID = ( *env )->GetMethodID( env,
                                                    vector_class,
                                                    "size",
                                                    "()I" );
            if ( vector_size_ID == NULL )
               errexit( "Could not get Vector.size()\n" );

               
            /* get size of Object vector */
            vector_size = ( int ) ( *env )->CallIntMethod( env,
                                            object_vector,
                                            vector_size_ID );
 
            sprintf( buf, "%d", vector_size - 1 );
            ( void ) write( slave_socket, buf, LINELEN );

            /* if this is final link in chain, call finalizer */
            if ( call_finalizer == TRUE )
            {
               /* get pjvm.collectGarbage() */
               collect_garbage_ID = ( *env )->GetMethodID( env,
                                              pjvm_cls,
                                              "collectGarbage",
                                              "()V" );
               if ( collect_garbage_ID == NULL )
                  errexit( "Could not get pjvm.collectGarbage().\n" );

               /* call collectGarbage() */
               ( *env )->CallVoidMethod( env, pjvm, collect_garbage_ID
                                       );
            }
         }

         /* reset argument count */
         arg_count = 0;
      }

      ( void ) close( slave_socket ); 
      instantiate_failed = JNI_FALSE;
      continue;
   }
   return 0;
}
