/* $Id: requestjvm.c,v 1.1 2001/10/23 21:30:02 eric Exp $ */

/* SUMMARY
 *
 * requestjvm.c - requestjvm
 *
 * The program requestjvm contacts the PJVM Manager, which in turn 
 * contacts the PJVM Server, and requests the creation of a new JVM and
 * the allocation of a new port at which the new JVM can be contacted.  
 * If successful, the Server returns that port number to the Manager,
 * which stores it for future communications with other client 
 * utilities.
 *
 *
 * REVISION HISTORY
 *
 * $Log: requestjvm.c,v $
 * Revision 1.1  2001/10/23 21:30:02  eric
 * Initial revision
 *
 */

#define _USE_BSD
#include <sys/types.h>
#include <sys/signal.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/errno.h>
#include <netinet/in.h>

#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include "connectTCP.h"
#include "errexit.h"
#include "strtok2.h"

#define QLEN    32
#define BUFSIZE 4096
#define LINELEN 128

extern int errno;

int requestjvm( const char *host, const char *service, int argc,
                char *arguments );


/*  Function main()
 *  --------------------------------------------------------------------
 *
 *    TCP client for PJVM service - request new JVM.
 *
 */

int main( int argc, char *argv[] )
{
   char host[ LINELEN + 1 ], service[ LINELEN + 1 ];
   int i, remaining_length = LINELEN - 4, c, arg_count;
   char arguments[ remaining_length + 1 ];

   strcpy( arguments, "" );
   strcpy( host, "localhost" );
   strcpy( service, "pjvmmanager" );

   /* disable getopt() error messages */
   opterr = 0;

   while ( ( c = getopt( argc, argv, "h:s:" ) ) != -1 )
   {
      switch ( c )
      {
         case 'h': if ( strlen( optarg ) != 0 )
                      strncpy( host, optarg, LINELEN );
                   break;
         case 's': if ( strlen( optarg ) != 0 )
                      strncpy( service, optarg, LINELEN );
                   break;
         case '?': errexit( "Usage: requestjvm [-h hostname] "
                            "[-s service] [-- JVM args]\n" );
      }
   }

   /* collect the arguments after "--" for the JVM */
   for ( i = optind; i < argc; i++ )
   {
      if ( remaining_length > strlen( argv[ i ] ) )
      {
         strcat( arguments, argv[ i ] );
         strcat( arguments, " " );
         remaining_length -= strlen( argv[ i ] + 1 );
      }
   }

   arg_count = i - optind;

   return requestjvm( host, service, arg_count, arguments );
}


/*  Function requestjvm()
 *  --------------------------------------------------------------------
 *
 *    Request a new JVM from PJVM daemon.
 *
 */

int requestjvm( const char *host, const char *service, int argc,
                char *arguments )
{
   char buf[ LINELEN + 1 ], *pointer = buf, delimiter;
   int s, n, inchars, jvm_number, port_number;

   s = connectTCP( host, service );
   
   if ( argc == 0 )
      sprintf( buf, "jvm %d", argc );
   else
      sprintf( buf, "jvm %d %s %c", argc, arguments, '\0' );
   ( void ) write( s, buf, LINELEN );

   for ( inchars = 0; inchars < LINELEN; inchars += n )
   {
      n = read( s, buf, LINELEN );
      if ( n < 0 )
         errexit( "read: %s\n", strerror( errno ) );
   }

   /* jvm_number and port_number need to be set separately, because
      printf evaluates its arguments last to first, which confuses
      strtok2() and causes the arguments to be printed in reverse
      order. */
   switch ( atoi( strtok2( &pointer, &delimiter, " " ) ) )
   {
      case  0:  jvm_number  = atoi( strtok2( 
                                    &pointer, &delimiter, " " ) );
                port_number = atoi( strtok2 (
                                    &pointer, &delimiter, " " ) );
                printf( "JVM %d created and listening on port %d.\n", 
                        jvm_number, port_number );
                break;
      case -1:  errexit( "Server out of memory.\n" );
      case -2:  errexit( "Could not create JVM.\n" );
      case -3:  errexit( "Could not find pjvm.class.\n" );
      case -4:  errexit( "Could not get constructor for class "
                         "pjvm.\n" );
      case -5:  errexit( "Could not instantiate pjvm.\n" );
      default:  errexit( "Error creating JVM.\n" );

   }
   return 0;
}

